<?php
require_once __DIR__ . '/../database.php';

$cache_geolocalizacion = [];

function procesarComando($mensaje, $usuario, $pdo) {
    $comando = strtolower(explode(' ', $mensaje)[0]);
    
    switch($comando) {
        case '/online':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? obtenerEstadoServidor($sqlserver) : 'Error de conexión al servidor';
            
        case '/setvip':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetVip($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/setvipf':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetVipForzado($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/setvipguild':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetVipGuild($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/setviprm':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetVipRm($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/setvipall':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetVipAll($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/setcoin':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetCoin($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/setcoinf':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetCoinForzado($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/setcoinrm':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetCoinRm($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/setcoinall':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetCoinAll($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/reset':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarReset($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/resetf':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarResetForzado($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/resetrm':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarResetRm($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/mreset':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarMasterReset($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/mresetf':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarMasterResetForzado($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/mresetrm':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarMasterResetRm($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/level':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarLevel($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/zen':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarZen($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/s':
        case '/a':
        case '/v':
        case '/e':
        case '/c':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarStats($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/setstats':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetStats($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/battlepass':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarBattlePass($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/battlerm':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarBattleRm($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/password':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarPassword($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/changepw':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarChangePw($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/info':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarInfo($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/infopj':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarInfoPj($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/infovip':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarInfoVip($sqlserver) : 'Error de conexión al servidor';
            
        case '/banacc':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarBanAcc($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/banchar':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarBanChar($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/banguild':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarBanGuild($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/ban':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarBan($mensaje, $sqlserver, $usuario) : 'Error de conexión al servidor';
            
        case '/unban':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarUnban($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/banco':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarBanco($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/top':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarTop($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/rastrear':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarRastrear($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/global':
            return procesarGlobal($mensaje, $pdo);
            
        case '/guildposts':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarGuildPosts($sqlserver) : 'Error de conexión al servidor';
            
        case '/whichguild':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarWhichGuild($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/fixonline':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarFixOnline($sqlserver) : 'Error de conexión al servidor';
            
        case '/clearlog':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarClearLog($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/index':
            return procesarIndex($mensaje);
            
        case '/indexrm':
            return procesarIndexRm($mensaje);
            
        case '/setsaldo':
            $sqlserver = conectarSQLServer($usuario);
            return $sqlserver ? procesarSetSaldo($mensaje, $sqlserver) : 'Error de conexión al servidor';
            
        case '/ayuda':
            return procesarAyuda();
            
        default:
            return '';
    }
}

// COMANDO /online - Estado del servidor y jugadores conectados
function obtenerEstadoServidor($sqlserver) {
    global $cache_geolocalizacion;
    
    try {
        $stmt = $sqlserver->prepare("SELECT IP FROM MEMB_STAT WHERE ConnectStat = 1");
        $stmt->execute();
        $jugadores = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        if (empty($jugadores)) {
            return "🎮 Estado del Servidor\n📊 No hay jugadores conectados";
        }
        
        $totalJugadores = count($jugadores);
        $ipsUnicas = array_unique($jugadores);
        $conexionesUnicas = count($ipsUnicas);
        $ipsAProcesar = array_slice($ipsUnicas, 0, 100);
        $paises = [];
        
        foreach ($ipsAProcesar as $ip) {
            if (!isset($cache_geolocalizacion[$ip])) {
                $cache_geolocalizacion[$ip] = obtenerPaisPorIP($ip);
            }
            
            $pais = $cache_geolocalizacion[$ip];
            if (isset($paises[$pais['nombre']])) {
                $paises[$pais['nombre']]['count']++;
            } else {
                $paises[$pais['nombre']] = ['count' => 1, 'emoji' => $pais['emoji']];
            }
        }
        
        arsort($paises);
        
        $respuesta = "🎮 Estado del Servidor\n📊 Resumen:\n👥 Jugadores conectados: $totalJugadores\n🌐 Conexiones únicas: $conexionesUnicas\n\n🌎 Distribución por País:\n";
        
        foreach ($paises as $nombre => $data) {
            $porcentaje = round(($data['count'] / count($ipsAProcesar)) * 100, 1);
            $respuesta .= "- $nombre {$data['emoji']}: {$data['count']} ($porcentaje%)\n";
        }
        
        if (count($ipsUnicas) > 100) {
            $respuesta .= "\n⚠️ Mostrando primeras 100 conexiones únicas";
        }
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al consultar el servidor";
    }
}

function obtenerPaisPorIP($ip) {
    if (preg_match('/^(192\.168\.|10\.|172\.(1[6-9]|2[0-9]|3[01])\.|127\.)/', $ip)) {
        return ['nombre' => 'Local', 'emoji' => '🏠'];
    }
    
    $context = stream_context_create(['http' => ['timeout' => 1, 'ignore_errors' => true]]);
    $response = @file_get_contents("https://api.country.is/$ip", false, $context);
    
    if ($response) {
        $data = json_decode($response, true);
        if ($data && isset($data['country'])) {
            $codigo = strtoupper($data['country']);
            $paises = [
                'DE' => ['nombre' => 'Alemania', 'emoji' => '🇩🇪'],
                'PE' => ['nombre' => 'Perú', 'emoji' => '🇵🇪'],
                'US' => ['nombre' => 'Estados Unidos', 'emoji' => '🇺🇸'],
                'AR' => ['nombre' => 'Argentina', 'emoji' => '🇦🇷'],
                'VE' => ['nombre' => 'Venezuela', 'emoji' => '🇻🇪'],
                'CO' => ['nombre' => 'Colombia', 'emoji' => '🇨🇴'],
                'MX' => ['nombre' => 'México', 'emoji' => '🇲🇽'],
                'BR' => ['nombre' => 'Brasil', 'emoji' => '🇧🇷'],
                'CL' => ['nombre' => 'Chile', 'emoji' => '🇨🇱'],
                'EC' => ['nombre' => 'Ecuador', 'emoji' => '🇪🇨'],
                'ES' => ['nombre' => 'España', 'emoji' => '🇪🇸'],
                'UY' => ['nombre' => 'Uruguay', 'emoji' => '🇺🇾'],
                'PY' => ['nombre' => 'Paraguay', 'emoji' => '🇵🇾'],
                'BO' => ['nombre' => 'Bolivia', 'emoji' => '🇧🇴'],
                'CR' => ['nombre' => 'Costa Rica', 'emoji' => '🇨🇷'],
                'PA' => ['nombre' => 'Panamá', 'emoji' => '🇵🇦']
            ];
            return $paises[$codigo] ?? ['nombre' => $codigo, 'emoji' => '🌍'];
        }
    }
    
    return ['nombre' => 'Desconocido', 'emoji' => '❓'];
}

function validarParametrosVip($parts, $esVipForzado = false) {
    $tipoComando = $esVipForzado ? "VIP Forzado" : "VIP";
    $formato = $esVipForzado ? "/setvipf cuenta(s) nivel días" : "/setvip cuenta nivel días";
    $ejemplos = $esVipForzado ? 
        "• Una cuenta: /setvipf MiCuenta 2 30\n• Varias cuentas: /setvipf user1,user2,user3 3 30" :
        "• Ejemplo: /setvip MiCuenta 2 30";
    $ventaja = $esVipForzado ? "\n⚡ Ventaja: Funciona aunque las cuentas estén conectadas\n✨ ¡Perfecto para activaciones masivas!" : "";
    
    $guia = "🚀 ¡Te explico el comando $tipoComando!\n\n📝 Formato: $formato\n\n🌟 Niveles disponibles:\n🥉 1: VIP Bronze\n🥈 2: VIP Silver\n🥇 3: VIP Gold\n\n💡 Ejemplos:\n$ejemplos$ventaja";
    
    if (count($parts) !== 4) return $guia;
    
    $level = intval($parts[2]);
    $days = intval($parts[3]);
    
    if ($level < 1 || $level > 3 || $days <= 0) {
        return "🤔 Te ayudo a ajustar los valores\n\n📋 Recuerda que:\n• El nivel debe ser 1, 2 o 3\n• Los días deben ser mayor a 0\n\n🌟 Niveles VIP:\n🥉 1: VIP Bronze\n🥈 2: VIP Silver\n🥇 3: VIP Gold\n\n✅ Ejemplo correcto: " . ($esVipForzado ? "/setvipf MiCuenta 2 30" : "/setvip MiCuenta 2 30");
    }
    
    return null;
}

function buscarCuenta($cuenta, $sqlserver) {
    $stmt = $sqlserver->prepare("SELECT ConnectStat FROM MEMB_STAT WHERE memb___id = ?");
    $stmt->execute([$cuenta]);
    $connect_stat = $stmt->fetchColumn();
    
    if ($connect_stat === false) {
        $stmt = $sqlserver->prepare("SELECT AccountID FROM Character WHERE Name = ?");
        $stmt->execute([$cuenta]);
        $account_from_char = $stmt->fetchColumn();
        
        if ($account_from_char) {
            $stmt = $sqlserver->prepare("SELECT ConnectStat FROM MEMB_STAT WHERE memb___id = ?");
            $stmt->execute([$account_from_char]);
            $connect_stat = $stmt->fetchColumn() ?: 0;
            return ['cuenta' => $account_from_char, 'conectado' => $connect_stat];
        }
        return false;
    }
    
    return ['cuenta' => $cuenta, 'conectado' => $connect_stat];
}

// COMANDO /setvip - Activar VIP (requiere desconexión)
function procesarSetVip($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    $validacion = validarParametrosVip($parts);
    if ($validacion) return $validacion;
    
    [$cmd, $cuenta, $level, $days] = $parts;
    $level = intval($level);
    $days = intval($days);
    
    try {
        $resultado = buscarCuenta($cuenta, $sqlserver);
        if (!$resultado) {
            return "🔍 No encontré esa cuenta\n\nNo pude localizar la cuenta o personaje '$cuenta'\n\n💡 Consejos:\n• Verifica que el nombre esté bien escrito\n• Puedes usar el nombre de cuenta o personaje\n• Revisa mayúsculas y minúsculas\n\n🔄 ¡Inténtalo de nuevo cuando tengas el nombre correcto!";
        }
        
        if ($resultado['conectado'] == 1) {
            return "⚠️ La cuenta está en uso\n\nLa cuenta '{$resultado['cuenta']}' está conectada al juego.\n\n🎯 Para activar VIP:\n1. Desconecta la cuenta del juego\n2. Vuelve a usar el comando\n\n💡 Esto es para evitar problemas con la activación.";
        }
        
        $stmt = $sqlserver->prepare("SELECT AccountExpireDate FROM MEMB_INFO WHERE memb___id = ?");
        $stmt->execute([$resultado['cuenta']]);
        $expire_date = $stmt->fetchColumn() ?: date('Y-m-d H:i:s');
        
        $currentExpireDateTime = new DateTime($expire_date);
        $currentExpireDateTime->modify("+$days days");
        $new_expire_date = $currentExpireDateTime->format('Y-m-d H:i:s');
        $display_date = $currentExpireDateTime->format('d/m/Y H:i:s');
        
        $vip_types = [1 => "VIP Bronze 🥉", 2 => "VIP Silver 🥈", 3 => "VIP Gold 🥇"];
        
        $stmt = $sqlserver->prepare("UPDATE MEMB_INFO SET AccountLevel = ?, AccountExpireDate = ? WHERE memb___id = ?");
        
        if ($stmt->execute([$level, $new_expire_date, $resultado['cuenta']])) {
            return "🎉 ¡VIP Activado Exitosamente!\n\n👤 Cuenta: {$resultado['cuenta']}\n🌟 Nivel: {$vip_types[$level]}\n⏳ Vence: $display_date";
        } else {
            return "😕 Algo salió mal\n\nNo pude activar el VIP para '{$resultado['cuenta']}' en este momento.\n\n🔄 Por favor inténtalo de nuevo en unos momentos.\n💬 Si el problema persiste, avísame.";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando VIP";
    }
}

// COMANDO /setvipf - Activar VIP sin desconexión (múltiples cuentas)
function procesarSetVipForzado($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje, 4);
    
    $validacion = validarParametrosVip($parts, true);
    if ($validacion) return $validacion;
    
    [$cmd, $cuentas, $level, $days] = $parts;
    $level = intval($level);
    $days = intval($days);
    
    $listaCuentas = array_map('trim', explode(',', $cuentas));
    $vipTypes = [1 => "VIP Bronze 🥉", 2 => "VIP Silver 🥈", 3 => "VIP Gold 🥇"];
    
    $exitosas = [];
    $fallidas = [];
    
    try {
        foreach ($listaCuentas as $cuenta) {
            $resultado = procesarCuentaVipForzado($cuenta, $level, $days, $sqlserver);
            $resultado['exito'] ? $exitosas[] = $resultado : $fallidas[] = $cuenta;
        }
        
        return generarRespuestaVipForzado($listaCuentas, $exitosas, $fallidas, $vipTypes[$level]);
        
    } catch (PDOException $e) {
        return "❌ Error VIP Forzado\nDetalles: " . $e->getMessage() . "\nContacta al administrador.";
    }
}

function procesarCuentaVipForzado($cuenta, $level, $days, $sqlserver) {
    $finalAccountId = $cuenta;
    $stmt = $sqlserver->prepare("SELECT memb___id FROM MEMB_INFO WHERE memb___id = ?");
    $stmt->execute([$cuenta]);
    
    if (!$stmt->fetchColumn()) {
        $stmt = $sqlserver->prepare("SELECT AccountID FROM Character WHERE Name = ?");
        $stmt->execute([$cuenta]);
        $accountFromChar = $stmt->fetchColumn();
        
        if (!$accountFromChar) return ['exito' => false];
        $finalAccountId = $accountFromChar;
    }
    
    $stmt = $sqlserver->prepare("SELECT AccountExpireDate FROM MEMB_INFO WHERE memb___id = ?");
    $stmt->execute([$finalAccountId]);
    $expireDate = $stmt->fetchColumn();
    
    if (!$expireDate) return ['exito' => false];
    
    $currentExpireDateTime = new DateTime($expireDate);
    $currentDate = new DateTime();
    
    $newDateTime = $currentExpireDateTime < $currentDate ? new DateTime() : clone $currentExpireDateTime;
    $newDateTime->modify("+$days days");
    
    $stmt = $sqlserver->prepare("UPDATE MEMB_INFO SET AccountLevel = ?, AccountExpireDate = ? WHERE memb___id = ?");
    
    if ($stmt->execute([$level, $newDateTime->format('Y-m-d H:i:s'), $finalAccountId])) {
        return [
            'exito' => true,
            'cuentaOriginal' => $cuenta,
            'cuentaFinal' => $finalAccountId,
            'fechaVencimiento' => $newDateTime->format('d/m/Y H:i:s')
        ];
    }
    
    return ['exito' => false];
}

function generarRespuestaVipForzado($listaCuentas, $exitosas, $fallidas, $tipoVip) {
    if (count($listaCuentas) === 1 && !empty($exitosas)) {
        $cuenta = $exitosas[0];
        $respuesta = "🎉 ¡VIP Forzado Activado Exitosamente!\n\n👤 Cuenta: {$cuenta['cuentaFinal']}\n";
        
        if ($cuenta['cuentaOriginal'] !== $cuenta['cuentaFinal']) {
            $respuesta .= "🎮 Personaje: {$cuenta['cuentaOriginal']}\n";
        }
        
        return $respuesta . "🌟 Nuevo Nivel: $tipoVip\n⏳ Vence: {$cuenta['fechaVencimiento']}\n⚡ Aplicado sin desconectar la cuenta\n💫 ¡Disfruta tus beneficios VIP!";
    }
    
    $respuesta = "";
    
    if (!empty($exitosas)) {
        $respuesta .= "🎉 ¡Activación VIP Masiva Exitosa!\n\n🌟 Nivel: $tipoVip\n⚡ Aplicado sin desconectar cuentas\n\n✨ Cuentas actualizadas:\n";
        
        foreach ($exitosas as $cuenta) {
            $respuesta .= "\n👤 {$cuenta['cuentaOriginal']}";
            if ($cuenta['cuentaOriginal'] !== $cuenta['cuentaFinal']) {
                $respuesta .= " (Cuenta: {$cuenta['cuentaFinal']})";
            }
            $respuesta .= "\n⏳ Vence: {$cuenta['fechaVencimiento']}\n";
        }
        
        $respuesta .= "\n🚀 ¡Todas las cuentas pueden seguir jugando!";
    }
    
    if (!empty($fallidas)) {
        if (!empty($respuesta)) $respuesta .= "\n\n";
        $respuesta .= "🔍 Algunas cuentas no se encontraron:\n";
        foreach ($fallidas as $cuenta) {
            $respuesta .= "• $cuenta\n";
        }
        $respuesta .= "\n💡 Verifica que los nombres estén escritos correctamente.";
    }
    
    if (empty($exitosas)) {
        return "🔍 No se encontraron las cuentas especificadas\n\n💡 Consejos:\n• Verifica que los nombres estén bien escritos\n• Puedes usar nombres de cuenta o personaje\n• Revisa mayúsculas y minúsculas\n• Para múltiples cuentas, sepáralas con comas\n\n🔄 ¡Inténtalo de nuevo cuando tengas los nombres correctos!";
    }
    
    return $respuesta;
}

// COMANDO /setvipguild - Activar VIP a todos los miembros de un guild
function procesarSetVipGuild($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje, 4);
    
    if (count($parts) !== 4) {
        return "📝 Formato: /setvipguild nombre_guild nivel días\n\n💡 Ejemplos:\n• /setvipguild UNITED 2 30\n• /setvipguild DRAGON 3 15\n\n🌟 Niveles VIP:\n🥉 1: VIP Bronze\n🥈 2: VIP Silver\n🥇 3: VIP Gold\n\nℹ️ Nota: Si el usuario ya tiene VIP igual o superior, solo se agregan días";
    }
    
    [$cmd, $guildName, $level, $days] = $parts;
    $level = intval($level);
    $days = intval($days);
    
    if ($level < 1 || $level > 3 || $days <= 0) {
        return "❌ Error de validación\n\n📋 Requisitos:\n• Nivel VIP: 1 a 3\n• Días: mayor a 0\n\n🔄 Intenta nuevamente";
    }
    
    try {
        $stmt = $sqlserver->prepare("SELECT COUNT(*) FROM GuildMember WHERE G_Name = ?");
        $stmt->execute([$guildName]);
        
        if ($stmt->fetchColumn() == 0) {
            return "❌ Guild no encontrado\n\n🏰 Guild: $guildName\n📝 Verifica que el nombre sea correcto";
        }
        
        $stmt = $sqlserver->prepare("SELECT Name FROM GuildMember WHERE G_Name = ?");
        $stmt->execute([$guildName]);
        $miembros = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        if (empty($miembros)) {
            return "❌ Sin miembros\n\n🏰 Guild: $guildName\n👥 No se encontraron miembros";
        }
        
        $vipTypes = [1 => "VIP Bronze 🥉", 2 => "VIP Silver 🥈", 3 => "VIP Gold 🥇"];
        $exitosos = [];
        $fallidos = [];
        $mejorados = [];
        $extendidos = [];
        
        foreach ($miembros as $personaje) {
            $resultado = procesarMiembroGuild($personaje, $level, $days, $sqlserver, $vipTypes);
            
            if ($resultado['exito']) {
                $exitosos[] = $resultado;
                $resultado['accion'] === 'upgrade' ? $mejorados[] = $resultado : $extendidos[] = $resultado;
            } else {
                $fallidos[] = $resultado['error'];
            }
        }
        
        return generarRespuestaGuild($guildName, $miembros, $exitosos, $fallidos, $mejorados, $extendidos, $vipTypes[$level], $days);
        
    } catch (PDOException $e) {
        return "Error al procesar el comando VIP de guild";
    }
}

function procesarMiembroGuild($personaje, $level, $days, $sqlserver, $vipTypes) {
    $stmt = $sqlserver->prepare("SELECT AccountID FROM Character WHERE Name = ?");
    $stmt->execute([$personaje]);
    $accountId = $stmt->fetchColumn();
    
    if (!$accountId) {
        return ['exito' => false, 'error' => "$personaje (personaje no encontrado)"];
    }
    
    $stmt = $sqlserver->prepare("SELECT AccountLevel, AccountExpireDate FROM MEMB_INFO WHERE memb___id = ?");
    $stmt->execute([$accountId]);
    $info = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$info) {
        return ['exito' => false, 'error' => "$personaje (cuenta no encontrada)"];
    }
    
    $currentLevel = intval($info['AccountLevel']);
    $currentExpire = $info['AccountExpireDate'];
    
    $newLevel = $level;
    $accion = "upgrade";
    
    if ($currentLevel >= $level && $currentLevel > 0) {
        $newLevel = $currentLevel;
        $accion = "extend";
    }
    
    $currentExpireDateTime = new DateTime($currentExpire);
    $currentDate = new DateTime();
    
    $newDateTime = $currentExpireDateTime < $currentDate ? new DateTime() : clone $currentExpireDateTime;
    $newDateTime->modify("+$days days");
    
    $stmt = $sqlserver->prepare("UPDATE MEMB_INFO SET AccountLevel = ?, AccountExpireDate = ? WHERE memb___id = ?");
    
    if ($stmt->execute([$newLevel, $newDateTime->format('Y-m-d H:i:s'), $accountId])) {
        return [
            'exito' => true,
            'personaje' => $personaje,
            'cuenta' => $accountId,
            'nivelAnterior' => $currentLevel,
            'nivelNuevo' => $newLevel,
            'fechaVencimiento' => $newDateTime->format('d/m/Y H:i:s'),
            'accion' => $accion
        ];
    }
    
    return ['exito' => false, 'error' => "$personaje (error al actualizar)"];
}

function generarRespuestaGuild($guildName, $miembros, $exitosos, $fallidos, $mejorados, $extendidos, $tipoVip, $days) {
    $totalMiembros = count($miembros);
    $exitososCount = count($exitosos);
    $fallidosCount = count($fallidos);
    $mejoradosCount = count($mejorados);
    $extendidosCount = count($extendidos);
    
    $respuesta = "🏰 ¡Actualización VIP de Guild!\n\n🏷️ Guild: $guildName\n🌟 Nivel Solicitado: $tipoVip\n📅 Duración: $days días\n\n📊 Resultados:\n✅ Exitosos: $exitososCount\n🆙 Nuevos/Mejorados: $mejoradosCount\n⏰ Extendidos: $extendidosCount\n❌ Fallidos: $fallidosCount\n👥 Total: $totalMiembros";
    
    if (!empty($fallidos)) {
        $respuesta .= "\n\n❌ Errores:\n";
        $mostrar = min(5, count($fallidos));
        for ($i = 0; $i < $mostrar; $i++) {
            $respuesta .= "• {$fallidos[$i]}\n";
        }
        if (count($fallidos) > 5) {
            $restantes = count($fallidos) - 5;
            $respuesta .= "... y $restantes más";
        }
    }
    
    return $respuesta;
}

// COMANDO /setviprm - Remover VIP de una cuenta
function procesarSetVipRm($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 2) {
        return "📝 Formato: /setviprm cuenta\n\n💡 Ejemplo:\n• /setviprm MiCuenta";
    }
    
    [$cmd, $cuenta] = $parts;
    
    try {
        $resultado = buscarCuenta($cuenta, $sqlserver);
        if (!$resultado) {
            return "🔍 No encontré esa cuenta\n\nNo pude localizar la cuenta o personaje '$cuenta'\n\n💡 Consejos:\n• Verifica que el nombre esté bien escrito\n• Puedes usar el nombre de cuenta o personaje\n• Revisa mayúsculas y minúsculas";
        }
        
        if ($resultado['conectado'] == 1) {
            return "⚠️ La cuenta está en uso\n\nLa cuenta '{$resultado['cuenta']}' está conectada al juego.\n\n🎯 Para remover VIP:\n1. Desconecta la cuenta del juego\n2. Vuelve a usar el comando";
        }
        
        $stmt = $sqlserver->prepare("SELECT AccountLevel FROM MEMB_INFO WHERE memb___id = ?");
        $stmt->execute([$resultado['cuenta']]);
        $accountLevel = $stmt->fetchColumn();
        
        if ($accountLevel < 1 || $accountLevel > 3) {
            return "❌ Sin VIP activo\n\nLa cuenta '{$resultado['cuenta']}' no tiene VIP activado";
        }
        
        $vipTypes = [1 => "VIP Bronze 🥉", 2 => "VIP Silver 🥈", 3 => "VIP Gold 🥇"];
        
        $stmt = $sqlserver->prepare("UPDATE MEMB_INFO SET AccountLevel = 0, AccountExpireDate = ? WHERE memb___id = ?");
        
        if ($stmt->execute([date('Y-m-d H:i:s'), $resultado['cuenta']])) {
            return "🎉 ¡VIP Removido Exitosamente!\n\n👤 Cuenta: {$resultado['cuenta']}\n🌟 Nivel anterior: {$vipTypes[$accountLevel]}\n⏳ Estado: VIP Desactivado";
        } else {
            return "😕 Algo salió mal\n\nNo pude remover el VIP de '{$resultado['cuenta']}' en este momento.\n\n🔄 Por favor inténtalo de nuevo en unos momentos.";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando de remoción VIP";
    }
}

// COMANDO /setvipall - Activar VIP a todas las cuentas sin VIP válido
function procesarSetVipAll($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "📝 Formato: /setvipall nivel días\n\n💡 Ejemplo:\n• /setvipall 2 30\n\n🌟 Niveles VIP:\n🥉 1: VIP Bronze\n🥈 2: VIP Silver\n🥇 3: VIP Gold";
    }
    
    [$cmd, $level, $days] = $parts;
    $level = intval($level);
    $days = intval($days);
    
    if ($level < 1 || $level > 3 || $days <= 0) {
        return "❌ Error de validación\n\n📋 Requisitos:\n• Nivel VIP: 1 a 3\n• Días: mayor a 0\n\n🔄 Intenta nuevamente";
    }
    
    try {
        $stmt = $sqlserver->prepare("SELECT memb___id, AccountLevel, AccountExpireDate FROM MEMB_INFO");
        $stmt->execute();
        $cuentas = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($cuentas)) {
            return "❌ No se encontraron cuentas en la base de datos";
        }
        
        $vipTypes = [1 => "VIP Bronze 🥉", 2 => "VIP Silver 🥈", 3 => "VIP Gold 🥇"];
        $currentDate = new DateTime();
        $newExpireDate = (new DateTime())->modify("+$days days")->format('Y-m-d H:i:s');
        
        $exitosos = 0;
        $saltados = 0;
        $totalCuentas = count($cuentas);
        
        $stmt = $sqlserver->prepare("UPDATE MEMB_INFO SET AccountLevel = ?, AccountExpireDate = ? WHERE memb___id = ?");
        
        foreach ($cuentas as $cuenta) {
            if ($cuenta['AccountExpireDate'] !== null) {
                $expireDate = new DateTime($cuenta['AccountExpireDate']);
                if ($expireDate > $currentDate) {
                    $saltados++;
                    continue;
                }
            }
            
            if ($stmt->execute([$level, $newExpireDate, $cuenta['memb___id']])) {
                $exitosos++;
            }
        }
        
        return "🎉 ¡VIP Activado Exitosamente!\n\n📊 Resumen:\n👥 Total de cuentas: $totalCuentas\n✅ Cuentas actualizadas: $exitosos\n⏩ Cuentas saltadas: $saltados\n🌟 Nivel aplicado: {$vipTypes[$level]}";
        
    } catch (PDOException $e) {
        return "Error al procesar el comando VIP masivo";
    }
}

// COMANDO /setcoin - Añadir monedas a una cuenta
function procesarSetCoin($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 4) {
        return "📝 Formato: /setcoin cuenta tipo cantidad\n\n💡 Ejemplo:\n• /setcoin user1 1 100\n\n💰 Tipos de Moneda:\n1: WCoinC\n2: WCoinP\n3: Goblin Points\n4: Ruud";
    }
    
    [$cmd, $cuenta, $coinType, $amount] = $parts;
    $coinType = intval($coinType);
    $amount = intval($amount);
    
    if ($coinType < 1 || $coinType > 4 || $amount <= 0) {
        return "❌ Error de validación\n\n📋 Requisitos:\n• Tipo de moneda: 1 a 4\n• Cantidad: mayor a 0\n\n🔄 Intenta nuevamente";
    }
    
    try {
        $resultado = buscarCuenta($cuenta, $sqlserver);
        if (!$resultado) {
            return "🔍 No encontré esa cuenta\n\nNo pude localizar la cuenta o personaje '$cuenta'\n\n💡 Consejos:\n• Verifica que el nombre esté bien escrito\n• Puedes usar el nombre de cuenta o personaje\n• Revisa mayúsculas y minúsculas";
        }
        
        if ($resultado['conectado'] == 1) {
            return "⚠️ La cuenta está en uso\n\nLa cuenta '{$resultado['cuenta']}' está conectada al juego.\n\n🎯 Para añadir monedas:\n1. Desconecta la cuenta del juego\n2. Vuelve a usar el comando";
        }
        
        $coinConfig = [
            1 => ['table' => 'CashShopData', 'column' => 'WCoinC', 'name' => 'WCoinC'],
            2 => ['table' => 'CashShopData', 'column' => 'WCoinP', 'name' => 'WCoinP'],
            3 => ['table' => 'CashShopData', 'column' => 'GoblinPoint', 'name' => 'Goblin Points'],
            4 => ['table' => 'Character', 'column' => 'RuudMoney', 'name' => 'Ruud']
        ];
        
        $config = $coinConfig[$coinType];
        
        // Verificar si RuudMoney existe, si no usar RuudToken
        if ($coinType == 4) {
            $stmt = $sqlserver->prepare("SELECT RuudMoney FROM Character WHERE AccountID = ? LIMIT 1");
            $stmt->execute([$resultado['cuenta']]);
            if ($stmt->errorCode() !== '00000') {
                $config['column'] = 'RuudToken';
            }
        }
        
        // Obtener saldo actual
        $stmt = $sqlserver->prepare("SELECT {$config['column']} FROM {$config['table']} WHERE AccountID = ?");
        $stmt->execute([$resultado['cuenta']]);
        $currentAmount = $stmt->fetchColumn();
        
        if ($currentAmount === false) {
            return "❌ Error de búsqueda\n\nNo se encontró información de monedas para la cuenta '{$resultado['cuenta']}'";
        }
        
        // Actualizar saldo
        $newAmount = $currentAmount + $amount;
        $stmt = $sqlserver->prepare("UPDATE {$config['table']} SET {$config['column']} = ? WHERE AccountID = ?");
        
        if ($stmt->execute([$newAmount, $resultado['cuenta']])) {
            return "🎉 ¡Monedas Añadidas Exitosamente!\n\n👤 Cuenta: {$resultado['cuenta']}\n💰 Moneda: {$config['name']}\n🔢 Saldo anterior: $currentAmount\n➕ Cantidad añadida: $amount\n💸 Nuevo saldo: $newAmount";
        } else {
            return "😕 Algo salió mal\n\nNo pude actualizar las monedas de '{$resultado['cuenta']}' en este momento.\n\n🔄 Por favor inténtalo de nuevo en unos momentos.";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando de monedas";
    }
}

// COMANDO /setcoinf - Añadir monedas forzado (múltiples cuentas, funciona aunque estén conectadas)
function procesarSetCoinForzado($mensaje, $sqlserver) {
    $startTime = microtime(true);
    $parts = preg_split('/\s+/', $mensaje, 4);
    
    if (count($parts) !== 4) {
        return "📝 Formato: /setcoinf cuenta(s) tipo cantidad\n\n💡 Ejemplos:\n• Una cuenta: /setcoinf user1 1 100\n• Varias cuentas: /setcoinf user1,user2,user3 2 1000\n\n💰 Tipos de Moneda:\n1: WCoinC\n2: WCoinP\n3: Goblin Points\n4: Ruud\n\n⚡ Ventaja: Funciona aunque las cuentas estén conectadas";
    }
    
    [$cmd, $cuentas, $coinType, $amount] = $parts;
    $coinType = intval($coinType);
    $amount = intval($amount);
    
    if ($coinType < 1 || $coinType > 4 || $amount <= 0) {
        return "❌ Error de validación\n\n📋 Requisitos:\n• Tipo de moneda: 1 a 4\n• Cantidad: mayor a 0\n\n🔄 Intenta nuevamente";
    }
    
    try {
        $searchTerms = array_map('trim', explode(',', $cuentas));
        
        $coinConfigs = [
            1 => ['name' => 'WCoinC', 'configs' => [['table' => 'CashShopData', 'column' => 'WCoinC']]],
            2 => ['name' => 'WCoinP', 'configs' => [['table' => 'CashShopData', 'column' => 'WCoinP']]],
            3 => ['name' => 'Goblin Points', 'configs' => [['table' => 'CashShopData', 'column' => 'GoblinPoint']]],
            4 => ['name' => 'Ruud', 'configs' => [['table' => 'Character', 'column' => 'RuudMoney'], ['table' => 'Character', 'column' => 'RuudToken']]]
        ];
        
        // Mega query para verificar estructura y resolver cuentas
        $placeholders = str_repeat('?,', count($searchTerms) - 1) . '?';
        $sqlMega = "
            SELECT 
                'structure' as type,
                CASE WHEN EXISTS(SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = 'CashShopData' AND COLUMN_NAME = 'WCoinC') THEN 1 ELSE 0 END as has_wcoinc,
                CASE WHEN EXISTS(SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = 'CashShopData' AND COLUMN_NAME = 'WCoinP') THEN 1 ELSE 0 END as has_wcoinp,
                CASE WHEN EXISTS(SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = 'CashShopData' AND COLUMN_NAME = 'GoblinPoint') THEN 1 ELSE 0 END as has_goblin,
                CASE WHEN EXISTS(SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = 'Character' AND COLUMN_NAME = 'RuudMoney') THEN 1 ELSE 0 END as has_ruud_money,
                CASE WHEN EXISTS(SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = 'Character' AND COLUMN_NAME = 'RuudToken') THEN 1 ELSE 0 END as has_ruud_token,
                '' as search_term, '' as account_id, 0 as current_amount
                
            UNION ALL
            
            SELECT 
                'account' as type, 0, 0, 0, 0, 0,
                mi.memb___id as search_term,
                mi.memb___id as account_id,
                0 as current_amount
            FROM MEMB_INFO mi 
            WHERE mi.memb___id IN ($placeholders)
            
            UNION ALL
            
            SELECT 
                'character' as type, 0, 0, 0, 0, 0,
                c.Name as search_term,
                c.AccountID as account_id,
                0 as current_amount
            FROM Character c 
            WHERE c.Name IN ($placeholders)
              AND c.Name NOT IN (SELECT memb___id FROM MEMB_INFO WHERE memb___id IN ($placeholders))";
        
        $params = array_merge($searchTerms, $searchTerms, $searchTerms);
        $stmtMega = $sqlserver->prepare($sqlMega);
        $stmtMega->execute($params);
        $megaResults = $stmtMega->fetchAll(PDO::FETCH_ASSOC);
        
        // Procesar resultados
        $structure = null;
        $accountMap = [];
        
        foreach ($megaResults as $row) {
            if ($row['type'] === 'structure') {
                $structure = $row;
            } else {
                $accountMap[$row['search_term']] = $row['account_id'];
            }
        }
        
        // Determinar configuración válida
        $validConfig = null;
        foreach ($coinConfigs[$coinType]['configs'] as $config) {
            $checkKey = match($config['table'] . '.' . $config['column']) {
                'CashShopData.WCoinC' => 'has_wcoinc',
                'CashShopData.WCoinP' => 'has_wcoinp',
                'CashShopData.GoblinPoint' => 'has_goblin',
                'Character.RuudMoney' => 'has_ruud_money',
                'Character.RuudToken' => 'has_ruud_token',
                default => null
            };
            
            if ($checkKey && $structure[$checkKey]) {
                $validConfig = $config;
                break;
            }
        }
        
        if (!$validConfig) {
            return "❌ Error de configuración\n\nLa moneda {$coinConfigs[$coinType]['name']} no está disponible en este servidor.";
        }
        
        // Obtener saldos actuales
        $validAccounts = array_filter($accountMap);
        if (empty($validAccounts)) {
            return generarRespuestaCoinForzado($searchTerms, [], $searchTerms, $coinConfigs[$coinType]['name'], $amount, 0);
        }
        
        $accountPlaceholders = str_repeat('?,', count($validAccounts) - 1) . '?';
        $sqlBalances = "SELECT AccountID, {$validConfig['column']} as current_amount 
                        FROM {$validConfig['table']} 
                        WHERE AccountID IN ($accountPlaceholders)";
        
        $stmtBalances = $sqlserver->prepare($sqlBalances);
        $stmtBalances->execute(array_values($validAccounts));
        $balances = $stmtBalances->fetchAll(PDO::FETCH_KEY_PAIR);
        
        // Preparar actualizaciones
        $successAccounts = [];
        $failedAccounts = [];
        $updates = [];
        
        foreach ($searchTerms as $searchTerm) {
            if (!isset($accountMap[$searchTerm])) {
                $failedAccounts[] = $searchTerm;
                continue;
            }
            
            $accountId = $accountMap[$searchTerm];
            $currentAmount = $balances[$accountId] ?? 0;
            $newAmount = $currentAmount + $amount;
            
            $successAccounts[] = [
                'id' => $searchTerm,
                'final_id' => $accountId,
                'current' => $currentAmount,
                'new' => $newAmount
            ];
            
            $updates[] = [$newAmount, $accountId];
        }
        
        // Ejecutar actualizaciones en transacción
        if (!empty($updates)) {
            $sqlserver->beginTransaction();
            try {
                $sqlUpdate = "UPDATE {$validConfig['table']} SET {$validConfig['column']} = ? WHERE AccountID = ?";
                $stmtUpdate = $sqlserver->prepare($sqlUpdate);
                
                foreach ($updates as $update) {
                    $stmtUpdate->execute($update);
                }
                
                $sqlserver->commit();
            } catch (Exception $e) {
                $sqlserver->rollBack();
                return "❌ Error en la actualización masiva. Operación cancelada.";
            }
        }
        
        $endTime = microtime(true);
        $executionTime = round(($endTime - $startTime) * 1000, 2);
        
        return generarRespuestaCoinForzado($searchTerms, $successAccounts, $failedAccounts, $coinConfigs[$coinType]['name'], $amount, $executionTime);
        
    } catch (PDOException $e) {
        return "❌ Error Monedas Forzado\nDetalles: " . $e->getMessage() . "\nContacta al administrador.";
    }
}

function generarRespuestaCoinForzado($searchTerms, $successAccounts, $failedAccounts, $coinName, $amount, $executionTime) {
    if (count($searchTerms) === 1) {
        if (!empty($successAccounts)) {
            $account = $successAccounts[0];
            $respuesta = "✅ Saldo actualizado exitosamente\n\n👤 Cuenta: {$account['final_id']}\n";
            if ($account['id'] !== $account['final_id']) {
                $respuesta .= "🎮 Personaje: {$account['id']}\n";
            }
            $respuesta .= "💰 Moneda: $coinName\n🔢 Saldo anterior: {$account['current']}\n➕ Cantidad añadida: $amount\n💸 Nuevo saldo: {$account['new']}\n⏱️ Tiempo de operación: {$executionTime} ms\n\n👍 ¡Operación completada!";
        } else {
            $respuesta = "🔍 No encontré esa cuenta\n\nNo pude localizar la cuenta o personaje '{$searchTerms[0]}'\n\n💡 Consejos:\n• Verifica que el nombre esté bien escrito\n• Puedes usar el nombre de cuenta o personaje\n• Revisa mayúsculas y minúsculas";
        }
    } else {
        $respuesta = "";
        if (!empty($successAccounts)) {
            $respuesta = "✅ Actualización de Saldo Exitosa\n\n💰 Moneda: $coinName\n➕ Cantidad añadida: $amount\n⏱️ Tiempo: {$executionTime}ms\n\n✨ Cuentas actualizadas:\n";
            foreach ($successAccounts as $account) {
                $respuesta .= "\n👤 {$account['id']}";
                if ($account['id'] !== $account['final_id']) {
                    $respuesta .= " (Cuenta: {$account['final_id']})";
                }
                $respuesta .= "\n💸 Saldo anterior: {$account['current']}\n💸 Nuevo saldo: {$account['new']}\n";
            }
        }
        
        if (!empty($failedAccounts)) {
            if (!empty($respuesta)) $respuesta .= "\n";
            $respuesta .= "🔍 Cuentas no encontradas:\n";
            foreach ($failedAccounts as $account) {
                $respuesta .= "• $account\n";
            }
            $respuesta .= "\n💡 Verifica que los nombres estén escritos correctamente.";
        }
        
        if (empty($successAccounts)) {
            $respuesta = "🔍 No se encontraron las cuentas especificadas\n\n💡 Consejos:\n• Verifica que los nombres estén bien escritos\n• Puedes usar nombres de cuenta o personaje\n• Para múltiples cuentas, sepáralas con comas\n\n🔄 ¡Inténtalo de nuevo cuando tengas los nombres correctos!";
        }
    }
    
    return $respuesta;
}

// COMANDO /setcoinrm - Remover monedas de cuentas
function procesarSetCoinRm($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje, 4);
    
    if (count($parts) !== 4) {
        return "📝 Formato: /setcoinrm cuenta(s) tipo cantidad\n\n💡 Ejemplos:\n• Una cuenta: /setcoinrm user1 1 100\n• Varias cuentas: /setcoinrm user1,user2,user3 2 500\n\n💰 Tipos de Moneda:\n1: WCoinC\n2: WCoinP\n3: Goblin Points\n4: Ruud";
    }
    
    [$cmd, $cuentas, $coinType, $amount] = $parts;
    $coinType = intval($coinType);
    $amount = intval($amount);
    
    if ($coinType < 1 || $coinType > 4 || $amount <= 0) {
        return "❌ Error de validación\n\n📋 Requisitos:\n• Tipo de moneda: 1 a 4\n• Cantidad: mayor a 0\n\n🔄 Intenta nuevamente";
    }
    
    try {
        $searchTerms = array_map('trim', explode(',', $cuentas));
        $coinConfig = [
            1 => ['table' => 'CashShopData', 'column' => 'WCoinC', 'name' => 'WCoinC'],
            2 => ['table' => 'CashShopData', 'column' => 'WCoinP', 'name' => 'WCoinP'],
            3 => ['table' => 'CashShopData', 'column' => 'GoblinPoint', 'name' => 'Goblin Points'],
            4 => ['table' => 'Character', 'column' => 'RuudMoney', 'name' => 'Ruud']
        ];
        
        $config = $coinConfig[$coinType];
        $successAccounts = [];
        $failedAccounts = [];
        
        foreach ($searchTerms as $searchTerm) {
            $finalAccountId = $searchTerm;
            
            // Buscar por cuenta
            $stmt = $sqlserver->prepare("SELECT memb___id FROM MEMB_INFO WHERE memb___id = ?");
            $stmt->execute([$searchTerm]);
            
            if (!$stmt->fetchColumn()) {
                // Buscar por personaje
                $stmt = $sqlserver->prepare("SELECT AccountID FROM Character WHERE Name = ?");
                $stmt->execute([$searchTerm]);
                $accountId = $stmt->fetchColumn();
                
                if (!$accountId) {
                    $failedAccounts[] = $searchTerm;
                    continue;
                }
                $finalAccountId = $accountId;
            }
            
            // Obtener saldo actual
            $stmt = $sqlserver->prepare("SELECT {$config['column']} FROM {$config['table']} WHERE AccountID = ?");
            $stmt->execute([$finalAccountId]);
            $currentAmount = $stmt->fetchColumn();
            
            // Si es Ruud y no existe RuudMoney, intentar RuudToken
            if ($coinType == 4 && ($currentAmount === false || $currentAmount < $amount)) {
                $stmt = $sqlserver->prepare("SELECT RuudToken FROM Character WHERE AccountID = ?");
                $stmt->execute([$finalAccountId]);
                $tokenAmount = $stmt->fetchColumn();
                
                if ($tokenAmount !== false && $tokenAmount >= $amount) {
                    $currentAmount = $tokenAmount;
                    $config['column'] = 'RuudToken';
                }
            }
            
            if ($currentAmount === false || $currentAmount < $amount) {
                $failedAccounts[] = $searchTerm;
                continue;
            }
            
            // Remover cantidad
            $newAmount = $currentAmount - $amount;
            $stmt = $sqlserver->prepare("UPDATE {$config['table']} SET {$config['column']} = ? WHERE AccountID = ?");
            
            if ($stmt->execute([$newAmount, $finalAccountId])) {
                $successAccounts[] = [
                    'id' => $searchTerm,
                    'final_id' => $finalAccountId,
                    'current' => $currentAmount,
                    'new' => $newAmount
                ];
            } else {
                $failedAccounts[] = $searchTerm;
            }
        }
        
        return generarRespuestaCoinRm($searchTerms, $successAccounts, $failedAccounts, $config['name'], $amount);
        
    } catch (PDOException $e) {
        return "Error al procesar el comando de remoción de monedas";
    }
}

function generarRespuestaCoinRm($searchTerms, $successAccounts, $failedAccounts, $coinName, $amount) {
    if (count($searchTerms) === 1) {
        if (!empty($successAccounts)) {
            $account = $successAccounts[0];
            $respuesta = "✅ Saldo actualizado exitosamente\n\n👤 Cuenta: {$account['final_id']}\n";
            if ($account['id'] !== $account['final_id']) {
                $respuesta .= "🎮 Personaje: {$account['id']}\n";
            }
            $respuesta .= "💰 Moneda: $coinName\n🔢 Saldo anterior: {$account['current']}\n➖ Cantidad descontada: $amount\n💸 Nuevo saldo: {$account['new']}";
        } else {
            $respuesta = "❌ Error de búsqueda\n\n👤 Usuario/Personaje: {$searchTerms[0]}\n🔎 No encontrado o saldo insuficiente";
        }
    } else {
        $respuesta = "";
        if (!empty($successAccounts)) {
            $respuesta = "✅ Descuento de Saldo Exitoso\n\n💰 Moneda: $coinName\n➖ Cantidad descontada: $amount\n\n✨ Cuentas actualizadas:\n";
            foreach ($successAccounts as $account) {
                $respuesta .= "\n👤 {$account['id']}";
                if ($account['id'] !== $account['final_id']) {
                    $respuesta .= " (Cuenta: {$account['final_id']})";
                }
                $respuesta .= "\n💸 Anterior: {$account['current']} → Nuevo: {$account['new']}\n";
            }
        }
        
        if (!empty($failedAccounts)) {
            if (!empty($respuesta)) $respuesta .= "\n";
            $respuesta .= "❌ Cuentas no encontradas o con saldo insuficiente:\n";
            foreach ($failedAccounts as $account) {
                $respuesta .= "• $account\n";
            }
        }
        
        if (empty($successAccounts)) {
            $respuesta = "❌ No se pudieron procesar las cuentas especificadas";
        }
    }
    
    return $respuesta;
}

// COMANDO /setcoinall - Asignar monedas a todas las cuentas
function procesarSetCoinAll($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "📝 Formato: /setcoinall tipo cantidad\n\n💡 Ejemplo:\n• /setcoinall 1 100\n\n💰 Tipos de Moneda:\n1: WCoinC\n2: WCoinP\n3: Goblin Points\n4: Ruud";
    }
    
    [$cmd, $coinType, $amount] = $parts;
    $coinType = intval($coinType);
    $amount = intval($amount);
    
    if ($coinType < 1 || $coinType > 4 || $amount <= 0) {
        return "❌ Error de validación\n\n📋 Requisitos:\n• Tipo de moneda: 1 a 4\n• Cantidad: mayor a 0\n\n🔄 Intenta nuevamente";
    }
    
    try {
        $coinConfig = [
            1 => ['table' => 'CashShopData', 'column' => 'WCoinC', 'name' => 'WCoinC'],
            2 => ['table' => 'CashShopData', 'column' => 'WCoinP', 'name' => 'WCoinP'],
            3 => ['table' => 'CashShopData', 'column' => 'GoblinPoint', 'name' => 'Goblin Points'],
            4 => ['table' => 'Character', 'column' => 'RuudMoney', 'name' => 'Ruud']
        ];
        
        $config = $coinConfig[$coinType];
        
        // Verificar si RuudMoney existe, si no usar RuudToken
        if ($coinType == 4) {
            $stmt = $sqlserver->prepare("SELECT RuudMoney FROM Character LIMIT 1");
            $stmt->execute();
            if ($stmt->errorCode() !== '00000') {
                $config['column'] = 'RuudToken';
            }
        }
        
        $stmt = $sqlserver->prepare("UPDATE {$config['table']} SET {$config['column']} = {$config['column']} + ?");
        
        if ($stmt->execute([$amount])) {
            return "✅ ¡Monedas asignadas exitosamente!\n\n💰 Moneda: {$config['name']}\n➕ Cantidad añadida: $amount\n👥 Aplicado a todas las cuentas";
        } else {
            return "❌ Error al actualizar las cuentas";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando de monedas masivo";
    }
}

// COMANDO /reset - Añadir resets (requiere desconectado)
function procesarReset($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /reset <Nombre> <Incremento>\n📌 Ejemplo: /reset Hero 1";
    }
    
    [$cmd, $charName, $amount] = $parts;
    $amount = intval($amount);
    
    if ($amount <= 0) {
        return "❌ El incremento debe ser mayor a 0";
    }
    
    try {
        $stmt = $sqlserver->prepare("SELECT AccountID, ResetCount FROM Character WHERE Name = ?");
        $stmt->execute([$charName]);
        $character = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$character) {
            return "❌ No se encontró el personaje '$charName'";
        }
        
        $accountId = $character['AccountID'];
        $currentResets = $character['ResetCount'];
        
        // Verificar conexión
        $stmt = $sqlserver->prepare("SELECT ConnectStat FROM MEMB_STAT WHERE memb___id = ?");
        $stmt->execute([$accountId]);
        $connectStat = $stmt->fetchColumn();
        
        if ($connectStat == 1) {
            return "⚠️ El personaje '$charName' está conectado\n🔒 Debe desconectarse para aplicar los cambios";
        }
        
        $newResets = $currentResets + $amount;
        $stmt = $sqlserver->prepare("UPDATE Character SET ResetCount = ? WHERE Name = ?");
        
        if ($stmt->execute([$newResets, $charName])) {
            return "✅ ¡Resets aplicados exitosamente!\n\n👤 Personaje: $charName\n🔄 Resets anteriores: $currentResets\n➕ Incremento: $amount\n🎯 Nuevo total: $newResets";
        } else {
            return "❌ Error al actualizar los resets";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando reset";
    }
}

// COMANDO /resetf - Añadir resets (funciona conectado)
function procesarResetForzado($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /resetf <Nombre> <Incremento>\n📌 Ejemplo: /resetf Hero 1\n💡 Funciona aunque el usuario esté conectado";
    }
    
    [$cmd, $charName, $amount] = $parts;
    $amount = intval($amount);
    
    if ($amount <= 0) {
        return "❌ El incremento debe ser mayor a 0";
    }
    
    try {
        $stmt = $sqlserver->prepare("SELECT ResetCount FROM Character WHERE Name = ?");
        $stmt->execute([$charName]);
        $currentResets = $stmt->fetchColumn();
        
        if ($currentResets === false) {
            return "❌ No se encontró el personaje '$charName'";
        }
        
        $newResets = $currentResets + $amount;
        $stmt = $sqlserver->prepare("UPDATE Character SET ResetCount = ? WHERE Name = ?");
        
        if ($stmt->execute([$newResets, $charName])) {
            return "✅ ¡Resets aplicados exitosamente!\n\n👤 Personaje: $charName\n🔄 Resets anteriores: $currentResets\n➕ Incremento: $amount\n🎯 Nuevo total: $newResets\n\n💡 Comando aplicado (usuario conectado permitido)";
        } else {
            return "❌ Error al actualizar los resets";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando reset forzado";
    }
}

// COMANDO /resetrm - Remover resets
function procesarResetRm($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /resetrm <Nombre> <Cantidad>\n📌 Ejemplo: /resetrm Hero 5\n💡 Remueve la cantidad especificada de resets";
    }
    
    [$cmd, $charName, $amount] = $parts;
    $amount = intval($amount);
    
    if ($amount <= 0) {
        return "❌ La cantidad a remover debe ser mayor a 0";
    }
    
    try {
        $stmt = $sqlserver->prepare("SELECT ResetCount FROM Character WHERE Name = ?");
        $stmt->execute([$charName]);
        $currentResets = $stmt->fetchColumn();
        
        if ($currentResets === false) {
            return "❌ No se encontró el personaje '$charName'";
        }
        
        if ($currentResets < $amount) {
            return "❌ Resets Insuficientes\n\n📊 Situación actual:\n🔸 Resets del personaje: $currentResets\n🔸 Cantidad a remover: $amount\n\n💡 No se puede remover más resets de los que tiene";
        }
        
        $newResets = max(0, $currentResets - $amount);
        $actualRemoved = $currentResets - $newResets;
        
        $stmt = $sqlserver->prepare("UPDATE Character SET ResetCount = ? WHERE Name = ?");
        
        if ($stmt->execute([$newResets, $charName])) {
            $respuesta = "✅ ¡Resets removidos exitosamente!\n\n👤 Personaje: $charName\n🔄 Resets anteriores: $currentResets\n➖ Cantidad removida: $actualRemoved\n🎯 Nuevo total: $newResets";
            
            if ($newResets === 0) {
                $respuesta .= "\n\n⚠️ ¡Atención! El personaje ha quedado sin resets";
            }
            
            return $respuesta;
        } else {
            return "❌ Error al actualizar los resets";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando de remoción de resets";
    }
}

// Función auxiliar para manejar master resets
function manejarMasterReset($charName, $amount, $operation, $checkConnection, $sqlserver) {
    try {
        $stmt = $sqlserver->prepare("SELECT AccountID, MasterResetCount FROM Character WHERE Name = ?");
        $stmt->execute([$charName]);
        $character = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$character) {
            return "❌ No se encontró el personaje '$charName'";
        }
        
        $accountId = $character['AccountID'];
        $currentCount = $character['MasterResetCount'];
        
        // Verificar conexión si es necesario
        if ($checkConnection) {
            $stmt = $sqlserver->prepare("SELECT ConnectStat FROM MEMB_STAT WHERE memb___id = ?");
            $stmt->execute([$accountId]);
            $connectStat = $stmt->fetchColumn();
            
            if ($connectStat == 1) {
                return "⚠️ El personaje '$charName' está conectado\n🔒 Debe desconectarse para aplicar los cambios";
            }
        }
        
        // Calcular nuevo valor
        if ($operation === 'remove') {
            if ($currentCount < $amount) {
                return "❌ Master Resets Insuficientes\n\n📊 Situación actual:\n🔸 Master Resets del personaje: $currentCount\n🔸 Cantidad a remover: $amount\n\n💡 No se puede remover más master resets de los que tiene";
            }
            $newCount = max(0, $currentCount - $amount);
            $action = "removidos";
            $changeText = "🔸 Cantidad removida: $amount";
        } else {
            $newCount = $currentCount + $amount;
            $action = "aplicados";
            $changeText = "🔸 Incremento: $amount";
        }
        
        $stmt = $sqlserver->prepare("UPDATE Character SET MasterResetCount = ? WHERE Name = ?");
        
        if (!$stmt->execute([$newCount, $charName])) {
            return "❌ Error al actualizar el MasterResetCount";
        }
        
        $respuesta = "✅ ¡Master Resets $action exitosamente!\n\n👤 Personaje: $charName\n🏆 Master Resets anteriores: $currentCount\n$changeText\n🎯 Nuevo total: $newCount";
        
        if ($operation === 'remove' && $newCount === 0) {
            $respuesta .= "\n\n⚠️ ¡Atención! El personaje ha quedado sin master resets";
        } else if ($operation === 'add' && !$checkConnection) {
            $respuesta .= "\n\n💡 Comando aplicado (usuario conectado permitido)";
        }
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando master reset";
    }
}

// COMANDO /mreset - Master resets (requiere desconectado)
function procesarMasterReset($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /mreset <Nombre> <Incremento>\n📌 Ejemplo: /mreset Hero 1";
    }
    
    [$cmd, $charName, $amount] = $parts;
    $amount = intval($amount);
    
    if ($amount <= 0) {
        return "❌ El incremento debe ser mayor a 0";
    }
    
    return manejarMasterReset($charName, $amount, 'add', true, $sqlserver);
}

// COMANDO /mresetf - Master resets (funciona conectado)
function procesarMasterResetForzado($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /mresetf <Nombre> <Incremento>\n📌 Ejemplo: /mresetf Hero 1\n💡 Funciona aunque el usuario esté conectado";
    }
    
    [$cmd, $charName, $amount] = $parts;
    $amount = intval($amount);
    
    if ($amount <= 0) {
        return "❌ El incremento debe ser mayor a 0";
    }
    
    return manejarMasterReset($charName, $amount, 'add', false, $sqlserver);
}

// COMANDO /mresetrm - Remover master resets
function procesarMasterResetRm($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /mresetrm <Nombre> <Cantidad>\n📌 Ejemplo: /mresetrm Hero 5\n💡 Remueve la cantidad especificada de master resets";
    }
    
    [$cmd, $charName, $amount] = $parts;
    $amount = intval($amount);
    
    if ($amount <= 0) {
        return "❌ La cantidad a remover debe ser mayor a 0";
    }
    
    return manejarMasterReset($charName, $amount, 'remove', true, $sqlserver);
}

// COMANDO /level - Modificar nivel de personaje
function procesarLevel($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /level <nombre> <nivel>\n📌 Ejemplo: /level Hero 400";
    }
    
    [$cmd, $charName, $level] = $parts;
    $level = intval($level);
    
    try {
        $stmt = $sqlserver->prepare("SELECT AccountID, cLevel FROM Character WHERE Name = ?");
        $stmt->execute([$charName]);
        $character = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$character) {
            return "❌ No se encontró el personaje '$charName'";
        }
        
        $accountId = $character['AccountID'];
        $currentLevel = $character['cLevel'];
        
        // Verificar conexión
        $stmt = $sqlserver->prepare("SELECT ConnectStat FROM MEMB_STAT WHERE memb___id = ?");
        $stmt->execute([$accountId]);
        $connectStat = $stmt->fetchColumn();
        
        if ($connectStat == 1) {
            return "⚠️ El personaje '$charName' está conectado\n\n❗ Debe desconectarse para aplicar los cambios";
        }
        
        $stmt = $sqlserver->prepare("UPDATE Character SET cLevel = ? WHERE Name = ?");
        
        if ($stmt->execute([$level, $charName])) {
            return "✅ ¡Nivel actualizado!\n\n👤 Personaje: $charName\n⭐ Nivel: $currentLevel ➜ $level";
        } else {
            return "❌ Error al actualizar el nivel";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando level";
    }
}

// COMANDO /zen - Añadir zen al personaje
function procesarZen($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /zen <Nombre> <Cantidad>\n📌 Ejemplo: /zen Hero 1000000";
    }
    
    [$cmd, $charName, $amount] = $parts;
    $amount = intval($amount);
    
    try {
        $stmt = $sqlserver->prepare("SELECT Money FROM Character WHERE Name = ?");
        $stmt->execute([$charName]);
        $currentMoney = $stmt->fetchColumn();
        
        if ($currentMoney === false) {
            return "❌ No se encontró el personaje '$charName'";
        }
        
        $maxMoney = 2000000000;
        $newMoney = $currentMoney + $amount;
        
        if ($newMoney > $maxMoney) {
            $newMoney = $maxMoney;
            $actualIncrement = $maxMoney - $currentMoney;
        } else {
            $actualIncrement = $amount;
        }
        
        $stmt = $sqlserver->prepare("UPDATE Character SET Money = ? WHERE Name = ?");
        
        if ($stmt->execute([$newMoney, $charName])) {
            return "✅ ¡Zen agregado exitosamente!\n\n👤 Personaje: $charName\n💰 Zen anterior: $currentMoney\n➕ Incremento: $actualIncrement\n💸 Nuevo total: $newMoney";
        } else {
            return "❌ Error al actualizar el zen";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando zen";
    }
}

// COMANDO /s, /a, /v, /e, /c - Stats individuales
function procesarStats($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\nUsa:\n/s <nombre> <valor> - Fuerza 💪\n/a <nombre> <valor> - Agilidad 🎯\n/v <nombre> <valor> - Vitalidad ❤️\n/e <nombre> <valor> - Energía ✨\n/c <nombre> <valor> - Command 👑\n\nEjemplo: /s LuisADM 65000";
    }
    
    [$cmd, $charName, $value] = $parts;
    $attribute = substr($cmd, 1); // Remover el /
    $value = intval($value);
    
    if ($value < 0) {
        return "❌ El valor debe ser un número positivo";
    }
    
    try {
        $stmt = $sqlserver->prepare("
            SELECT C.AccountID, C.Class, C.Strength, C.Dexterity, C.Vitality, C.Energy, C.LeaderShip, M.ConnectStat
            FROM Character C
            LEFT JOIN MEMB_STAT M ON C.AccountID = M.memb___id
            WHERE C.Name = ?
        ");
        $stmt->execute([$charName]);
        $character = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$character) {
            return "❌ El personaje '$charName' no fue encontrado";
        }
        
        if ($character['ConnectStat'] == 1) {
            return "⚠️ El personaje '$charName' está conectado\n\nDebe desconectarse para aplicar los cambios";
        }
        
        $fieldInfo = [
            's' => ['field' => 'Strength', 'name' => 'Fuerza', 'emoji' => '💪'],
            'a' => ['field' => 'Dexterity', 'name' => 'Agilidad', 'emoji' => '🎯'],
            'v' => ['field' => 'Vitality', 'name' => 'Vitalidad', 'emoji' => '❤️'],
            'e' => ['field' => 'Energy', 'name' => 'Energía', 'emoji' => '✨'],
            'c' => ['field' => 'LeaderShip', 'name' => 'Command', 'emoji' => '👑']
        ];
        
        if (!isset($fieldInfo[$attribute])) {
            return "❌ Atributo inválido";
        }
        
        $field = $fieldInfo[$attribute];
        
        // Verificar si es Dark Lord para Command
        if ($attribute === 'c' && $character['Class'] < 64) {
            return "❌ Solo los Dark Lord pueden tener Command";
        }
        
        $stmt = $sqlserver->prepare("UPDATE Character SET {$field['field']} = ? WHERE Name = ?");
        
        if ($stmt->execute([$value, $charName])) {
            $respuesta = "✅ ¡Estadística actualizada!\n👤 Personaje: $charName\n{$field['emoji']} {$field['name']}: {$character[$field['field']]} ➜ $value\n\n📊 Estadísticas actuales:\n";
            $respuesta .= "💪 Fuerza: " . ($attribute === 's' ? $value : $character['Strength']) . "\n";
            $respuesta .= "🎯 Agilidad: " . ($attribute === 'a' ? $value : $character['Dexterity']) . "\n";
            $respuesta .= "❤️ Vitalidad: " . ($attribute === 'v' ? $value : $character['Vitality']) . "\n";
            $respuesta .= "✨ Energía: " . ($attribute === 'e' ? $value : $character['Energy']) . "\n";
            if ($character['Class'] >= 64) {
                $respuesta .= "👑 Command: " . ($attribute === 'c' ? $value : $character['LeaderShip']) . "\n";
            }
            
            return $respuesta;
        } else {
            return "❌ Error al actualizar la estadística";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando de stats";
    }
}

// COMANDO /setstats - Todas las stats al mismo valor
function procesarSetStats($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /setstats <nombre> <valor>\n📌 Ejemplo: /setstats LuisADM 65000\n\n💡 Establece todos los atributos al mismo valor";
    }
    
    [$cmd, $charName, $value] = $parts;
    $value = intval($value);
    
    if ($value < 0) {
        return "❌ El valor debe ser un número positivo";
    }
    
    try {
        $stmt = $sqlserver->prepare("
            SELECT C.AccountID, C.Class, C.Strength, C.Dexterity, C.Vitality, C.Energy, C.LeaderShip, M.ConnectStat
            FROM Character C
            LEFT JOIN MEMB_STAT M ON C.AccountID = M.memb___id
            WHERE C.Name = ?
        ");
        $stmt->execute([$charName]);
        $character = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$character) {
            return "❌ El personaje '$charName' no fue encontrado";
        }
        
        if ($character['ConnectStat'] == 1) {
            return "⚠️ El personaje '$charName' está conectado\n\nDebe desconectarse para aplicar los cambios";
        }
        
        $isDarkLord = $character['Class'] >= 64;
        
        $sqlUpdate = "UPDATE Character SET 
            Strength = ?,
            Dexterity = ?,
            Vitality = ?,
            Energy = ?";
        
        $params = [$value, $value, $value, $value];
        
        if ($isDarkLord) {
            $sqlUpdate .= ", LeaderShip = ?";
            $params[] = $value;
        }
        
        $sqlUpdate .= " WHERE Name = ?";
        $params[] = $charName;
        
        $stmt = $sqlserver->prepare($sqlUpdate);
        
        if ($stmt->execute($params)) {
            $respuesta = "✅ ¡Todas las estadísticas actualizadas!\n\n👤 Personaje: $charName\n\n📊 Cambios realizados:\n";
            $respuesta .= "💪 Fuerza: {$character['Strength']} ➜ $value\n";
            $respuesta .= "🎯 Agilidad: {$character['Dexterity']} ➜ $value\n";
            $respuesta .= "❤️ Vitalidad: {$character['Vitality']} ➜ $value\n";
            $respuesta .= "✨ Energía: {$character['Energy']} ➜ $value\n";
            
            if ($isDarkLord) {
                $respuesta .= "👑 Command: {$character['LeaderShip']} ➜ $value\n";
            }
            
            $respuesta .= "\n✨ Todos los atributos han sido establecidos en $value";
            
            return $respuesta;
        } else {
            return "❌ Error al actualizar las estadísticas";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando setstats";
    }
}

// COMANDO /battlepass - Añadir estrellas de battle pass
function procesarBattlePass($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /battlepass <nombre> <estrellas>\n📌 Ejemplo: /battlepass 1HIT 300\n\n💡 Añade estrellas al Battle Pass del personaje";
    }
    
    [$cmd, $charName, $stars] = $parts;
    $stars = intval($stars);
    
    if ($stars <= 0) {
        return "❌ Error de validación\n\n📋 Requisitos:\n• Cantidad de estrellas debe ser mayor a 0\n\n🔄 Intenta nuevamente con un valor válido";
    }
    
    try {
        $stmt = $sqlserver->prepare("SELECT StarCount FROM CustomBattlePassData WHERE Name = ?");
        $stmt->execute([$charName]);
        $currentStars = $stmt->fetchColumn();
        
        if ($currentStars === false) {
            return "❌ Error de búsqueda\n\n👤 Personaje: $charName\n📝 Estado: No encontrado en la base de datos\n\n💡 Verifica el nombre e intenta nuevamente";
        }
        
        $newStars = $currentStars + $stars;
        $stmt = $sqlserver->prepare("UPDATE CustomBattlePassData SET StarCount = ? WHERE Name = ?");
        
        if ($stmt->execute([$newStars, $charName])) {
            $respuesta = "✅ ¡Battle Pass Actualizado!\n\n👤 Personaje: $charName\n⭐ Estrellas anteriores: $currentStars\n📈 Estrellas añadidas: $stars\n🌟 Total de estrellas: $newStars";
            
            if ($newStars >= 1000) {
                $respuesta .= "\n\n🏆 ¡Felicitaciones! Has alcanzado un hito importante";
            } elseif ($newStars >= 500) {
                $respuesta .= "\n\n🎯 ¡Buen progreso! Continúa así";
            }
            
            return $respuesta;
        } else {
            return "❌ Error al actualizar la base de datos";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando battlepass";
    }
}

// COMANDO /battlerm - Remover estrellas de battle pass
function procesarBattleRm($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato incorrecto\n\n📝 Uso: /battlerm <nombre> <estrellas>\n📌 Ejemplo: /battlerm 1HIT 300\n\n💡 Reduce las estrellas del Battle Pass del personaje";
    }
    
    [$cmd, $charName, $stars] = $parts;
    $stars = intval($stars);
    
    if ($stars <= 0) {
        return "❌ Error de validación\n\n📋 Requisitos:\n• Cantidad de estrellas debe ser mayor a 0\n\n🔄 Intenta nuevamente con un valor válido";
    }
    
    try {
        $stmt = $sqlserver->prepare("SELECT StarCount FROM CustomBattlePassData WHERE Name = ?");
        $stmt->execute([$charName]);
        $currentStars = $stmt->fetchColumn();
        
        if ($currentStars === false) {
            return "❌ Error de búsqueda\n\n👤 Personaje: $charName\n📝 Estado: No encontrado en la base de datos\n\n💡 Verifica el nombre e intenta nuevamente";
        }
        
        $newStars = max(0, $currentStars - $stars);
        $stmt = $sqlserver->prepare("UPDATE CustomBattlePassData SET StarCount = ? WHERE Name = ?");
        
        if ($stmt->execute([$newStars, $charName])) {
            $respuesta = "✅ ¡Estrellas removidas exitosamente!\n\n👤 Personaje: $charName\n⭐ Estrellas anteriores: $currentStars\n📉 Estrellas removidas: $stars\n🌟 Estrellas actuales: $newStars";
            
            if ($newStars === 0) {
                $respuesta .= "\n\n⚠️ ¡Atención! El personaje ha quedado sin estrellas";
            }
            
            return $respuesta;
        } else {
            return "❌ Error al actualizar la base de datos";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando battlerm";
    }
}

// COMANDO /password - Obtener contraseña de cuenta
function procesarPassword($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 2) {
        return "🚫 Formato incorrecto\n\n📝 Uso: /password <memb___id>\n\nEjemplo: /password tamiel";
    }
    
    [$cmd, $membId] = $parts;
    
    try {
        $stmt = $sqlserver->prepare("SELECT memb___id, memb__pwd, mail_addr FROM MEMB_INFO WHERE memb___id = ?");
        $stmt->execute([$membId]);
        $userInfo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$userInfo) {
            return "🚫 No se encontró una cuenta con el ID '$membId'";
        }
        
        return "✅ Información de cuenta recuperada\n\n🆔 Usuario: {$userInfo['memb___id']}\n🔑 Contraseña: {$userInfo['memb__pwd']}\n📧 Correo: {$userInfo['mail_addr']}\n\n🔒 Guarda esta información en un lugar seguro";
        
    } catch (PDOException $e) {
        return "Error al procesar el comando password";
    }
}

// COMANDO /changepw - Cambiar contraseña
function procesarChangePw($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    $totalParts = count($parts);
    
    if ($totalParts < 2 || $totalParts > 3) {
        return "🚫 Formato incorrecto\n\n📝 Uso: /changepw <memb___id> [dificultad 1-5 | contraseña_personalizada]\n\nEjemplos:\n/changepw tamiel\n/changepw tamiel 3\n/changepw tamiel MiPassword123!";
    }
    
    [$cmd, $membId] = $parts;
    $customPassword = null;
    $difficulty = 3;
    
    if (isset($parts[2])) {
        $secondParam = trim($parts[2]);
        
        if (is_numeric($secondParam) && intval($secondParam) >= 1 && intval($secondParam) <= 5) {
            $difficulty = intval($secondParam);
        } else {
            $customPassword = $secondParam;
            
            if (strlen($customPassword) < 6) {
                return "⚠️ Contraseña muy corta\n\nLa contraseña debe tener al menos 6 caracteres\n\n📝 Ejemplo: /changepw $membId MiPassword123!";
            }
            
            if (strlen($customPassword) > 50) {
                return "⚠️ Contraseña muy larga\n\nLa contraseña no puede exceder 50 caracteres";
            }
        }
    }
    
    try {
        $stmt = $sqlserver->prepare("SELECT memb___id FROM MEMB_INFO WHERE memb___id = ?");
        $stmt->execute([$membId]);
        
        if (!$stmt->fetch()) {
            return "🚫 No se encontró una cuenta con el ID '$membId'";
        }
        
        if ($customPassword !== null) {
            $newPassword = $customPassword;
            $passwordType = "personalizada";
            $securityInfo = "🔧 Contraseña personalizada";
        } else {
            // Generar contraseña automática
            $lowercase = 'abcdefghijklmnopqrstuvwxyz';
            $uppercase = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
            $numbers = '0123456789';
            $symbols = '!@#$%&*-_=+';
            $complexSymbols = '!@#$%&*-_=+^~?.,:;(){}[]<>/\\|\'"`';
            
            $length = 8 + $difficulty;
            
            switch ($difficulty) {
                case 1:
                    $chars = $lowercase . $uppercase . $numbers;
                    break;
                case 2:
                    $chars = $lowercase . $uppercase . $numbers . $symbols;
                    break;
                case 3:
                    $chars = $lowercase . $uppercase . $numbers . $symbols;
                    break;
                case 4:
                    $chars = $lowercase . $uppercase . $numbers . $symbols . $complexSymbols;
                    break;
                case 5:
                    $chars = $lowercase . $uppercase . $numbers . $symbols . $complexSymbols;
                    break;
            }
            
            $newPassword = '';
            $newPassword .= $lowercase[rand(0, strlen($lowercase) - 1)];
            $newPassword .= $uppercase[rand(0, strlen($uppercase) - 1)];
            $newPassword .= $numbers[rand(0, strlen($numbers) - 1)];
            
            if ($difficulty >= 2) {
                $newPassword .= $symbols[rand(0, strlen($symbols) - 1)];
            }
            
            if ($difficulty >= 4) {
                $newPassword .= $complexSymbols[rand(0, strlen($complexSymbols) - 1)];
            }
            
            while (strlen($newPassword) < $length) {
                $newPassword .= $chars[rand(0, strlen($chars) - 1)];
            }
            
            $newPassword = str_shuffle($newPassword);
            $passwordType = "automática";
            $difficultyText = str_repeat("★", $difficulty) . str_repeat("☆", 5 - $difficulty);
            $securityInfo = "🔒 Nivel de seguridad: $difficultyText";
        }
        
        $stmt = $sqlserver->prepare("UPDATE MEMB_INFO SET memb__pwd = ? WHERE memb___id = ?");
        
        if ($stmt->execute([$newPassword, $membId])) {
            return "✅ Contraseña actualizada exitosamente\n\n🆔 Usuario: $membId\n🔑 Nueva contraseña: $newPassword\n🎯 Tipo: Contraseña $passwordType\n$securityInfo\n\nGuarda esta información en un lugar seguro";
        } else {
            return "⚠️ Hubo un error al intentar cambiar la contraseña";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando changepw";
    }
}

// COMANDO /info - Información de cuenta
function procesarInfo($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 2) {
        return "❌ Comando incorrecto\n\n📝 Uso: /info <usuario/personaje>\n\nEjemplos:\n/info Usuario1\n/info Personaje1";
    }
    
    [$cmd, $searchTerm] = $parts;
    
    $getVIPRank = fn($level) => match($level) {
        1 => "🥉 VIP Bronze",
        2 => "🥈 VIP Silver", 
        3 => "🥇 VIP Gold",
        default => "Sin VIP"
    };
    
    try {
        // Buscar por ID
        $stmt = $sqlserver->prepare("
            SELECT mi.memb___id, mi.memb__pwd, mi.AccountLevel, mi.AccountExpireDate, ms.IP
            FROM MEMB_INFO mi
            LEFT JOIN MEMB_STAT ms ON mi.memb___id = ms.memb___id
            WHERE mi.memb___id = ?
        ");
        $stmt->execute([$searchTerm]);
        $userInfo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Si no existe, buscar por personaje
        if (!$userInfo) {
            $stmt = $sqlserver->prepare("
                SELECT mi.memb___id, mi.memb__pwd, mi.AccountLevel, mi.AccountExpireDate, ms.IP
                FROM Character c
                INNER JOIN MEMB_INFO mi ON c.AccountID = mi.memb___id
                LEFT JOIN MEMB_STAT ms ON mi.memb___id = ms.memb___id
                WHERE c.Name = ?
            ");
            $stmt->execute([$searchTerm]);
            $userInfo = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        if (!$userInfo) {
            return "🚫 No se encontró la cuenta ni el personaje '$searchTerm'";
        }
        
        $membId = $userInfo['memb___id'];
        $password = $userInfo['memb__pwd'];
        $expireDate = date('d/m/Y', strtotime($userInfo['AccountExpireDate']));
        $vipRank = $getVIPRank($userInfo['AccountLevel']);
        
        $respuesta = "📋 INFORMACIÓN DE CUENTA\n\n👤 Usuario: $membId\n🔑 Contraseña: $password\n📊 Nivel de VIP: $vipRank\n📅 Vence: $expireDate\n";
        
        if (!empty($userInfo['IP'])) {
            $respuesta .= "🌐 IP: {$userInfo['IP']}\n";
        }
        
        // Obtener monedas
        try {
            $stmt = $sqlserver->prepare("SELECT WCoinC, WCoinP, GoblinPoint FROM CashShopData WHERE AccountID = ?");
            $stmt->execute([$membId]);
            $coins = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($coins) {
                $respuesta .= "\n💰 MONEDAS\n\n🪙 WCoinC: {$coins['WCoinC']}\n🪙 WCoinP: {$coins['WCoinP']}\n🪙 GoblinPoint: {$coins['GoblinPoint']}\n";
            }
        } catch (PDOException $e) {
            // Tabla no existe, continuar
        }
        
        // Obtener personajes
        $stmt = $sqlserver->prepare("SELECT Name, ResetCount FROM Character WHERE AccountID = ? ORDER BY Name");
        $stmt->execute([$membId]);
        $characters = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if ($characters) {
            $respuesta .= "\n⚔️ PERSONAJES\n\n";
            foreach ($characters as $char) {
                $respuesta .= "⚔️ {$char['Name']}\n🔄 Resets: {$char['ResetCount']}\n\n";
            }
        }
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando info";
    }
}

// COMANDO /infopj - Información del personaje
function procesarInfoPj($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 2) {
        return "❌ Formato incorrecto\n\n📝 Uso: /infopj <nombre>\n\nEjemplo: /infopj LuisADM";
    }
    
    [$cmd, $charName] = $parts;
    
    try {
        $stmt = $sqlserver->prepare("
            SELECT 
                C.Name, C.Class, C.cLevel as Level, C.LevelUpPoint, C.ResetCount, C.Money,
                C.MapNumber, C.MapPosX, C.MapPosY, C.AccountID, M.ConnectStat,
                C.Strength, C.Dexterity, C.Vitality, C.Energy, C.LeaderShip
            FROM Character C
            LEFT JOIN MEMB_STAT M ON C.AccountID = M.memb___id
            WHERE C.Name = ?
        ");
        $stmt->execute([$charName]);
        $character = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$character) {
            return "❌ El personaje '$charName' no fue encontrado";
        }
        
        $classNames = [
            0 => "Dark Wizard", 1 => "Soul Master", 2 => "Grand Master",
            16 => "Dark Knight", 17 => "Blade Knight", 18 => "Blade Master",
            32 => "Fairy Elf", 33 => "Muse Elf", 34 => "High Elf",
            48 => "Magic Gladiator", 49 => "Duel Master",
            64 => "Dark Lord", 65 => "Lord Emperor"
        ];
        
        $mapNames = [
            0 => "Lorencia", 1 => "Dungeon", 2 => "Devias", 3 => "Noria",
            4 => "Lost Tower", 6 => "Arena", 7 => "Atlans", 8 => "Tarkan",
            33 => "Aida", 34 => "Crywolf", 51 => "Elbeland", 63 => "Vulcanus"
        ];
        
        $className = $classNames[$character['Class']] ?? "Desconocida";
        $mapName = $mapNames[$character['MapNumber']] ?? "Mapa {$character['MapNumber']}";
        $status = $character['ConnectStat'] == 1 ? "🟢 Conectado" : "🔴 Desconectado";
        
        $respuesta = "📊 Información del Personaje\n\n👤 Nombre: {$character['Name']}\n💠 Clase: $className\n⭐ Nivel: {$character['Level']}\n🔶 Puntos: {$character['LevelUpPoint']}\n🔄 Resets: {$character['ResetCount']}\n💰 Zen: " . number_format($character['Money']) . "\n🌍 Ubicación: $mapName (X: {$character['MapPosX']}, Y: {$character['MapPosY']})\n$status\n\n📋 Estadísticas:\n💪 Fuerza: {$character['Strength']}\n🎯 Agilidad: {$character['Dexterity']}\n❤️ Vitalidad: {$character['Vitality']}\n✨ Energía: {$character['Energy']}\n";
        
        if ($character['Class'] >= 64) {
            $respuesta .= "👑 Command: {$character['LeaderShip']}\n";
        }
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando infopj";
    }
}

// COMANDO /infovip - Estadísticas de usuarios VIP
function procesarInfoVip($sqlserver) {
    try {
        $vipCounts = [1 => 0, 2 => 0, 3 => 0];
        $latestExpireDate = null;
        $latestAccountId = null;
        
        $stmt = $sqlserver->prepare("SELECT memb___id, AccountLevel, AccountExpireDate FROM MEMB_INFO WHERE AccountLevel IN (1, 2, 3) AND AccountExpireDate IS NOT NULL");
        $stmt->execute();
        $vipAccounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $hasActiveAccounts = false;
        
        foreach ($vipAccounts as $account) {
            $level = intval($account['AccountLevel']);
            
            $expireDate = null;
            if (!empty($account['AccountExpireDate'])) {
                $expireDate = DateTime::createFromFormat('d/m/Y H:i:s', $account['AccountExpireDate']);
                
                if (!$expireDate) {
                    $expireDate = DateTime::createFromFormat('Y-m-d H:i:s', $account['AccountExpireDate']);
                }
            }
            
            if ($expireDate && $expireDate > new DateTime()) {
                $hasActiveAccounts = true;
                
                if (isset($vipCounts[$level])) {
                    $vipCounts[$level]++;
                }
                
                if ($latestExpireDate === null || $expireDate > $latestExpireDate) {
                    $latestExpireDate = $expireDate;
                    $latestAccountId = $account['memb___id'];
                }
            }
        }
        
        $respuesta = "📊 Estadísticas de VIP\n\n🥉 VIP Bronze: {$vipCounts[1]}\n🥈 VIP Silver: {$vipCounts[2]}\n🥇 VIP Gold: {$vipCounts[3]}\n\n";
        
        if ($hasActiveAccounts && $latestExpireDate && $latestAccountId) {
            $formattedDate = $latestExpireDate->format('d/m/Y H:i:s');
            $respuesta .= "📅 Fecha de vencimiento más lejana:\n$formattedDate - $latestAccountId";
        } else {
            if (array_sum($vipCounts) > 0) {
                $respuesta .= "No hay cuentas VIP con fechas de vencimiento válidas";
            } else {
                $respuesta .= "No hay cuentas VIP activas";
            }
        }
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando infovip";
    }
}

// COMANDO /banacc - Banear/desbanear cuenta
function procesarBanAcc($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Comando incorrecto\n\n📝 Uso: /banacc <cuenta> <valor>\n\nEjemplos:\n/banacc Usuario1 1 (banear)\n/banacc Usuario1 0 (desbanear)";
    }
    
    [$cmd, $accountName, $banValue] = $parts;
    $banValue = intval($banValue);
    
    if ($banValue !== 0 && $banValue !== 1) {
        return "❌ Valor de ban inválido\n\nDebe ser 0 (desbanear) o 1 (banear)";
    }
    
    try {
        $stmt = $sqlserver->prepare("UPDATE MEMB_INFO SET bloc_code = ? WHERE memb___id = ?");
        $stmt->execute([$banValue, $accountName]);
        
        if ($stmt->rowCount() > 0) {
            $status = $banValue === 1 ? "baneada" : "desbaneada";
            return "✅ Acción exitosa\n\n👤 Cuenta: $accountName\n📊 Estado: $status";
        } else {
            return "❌ Error\n\nNo se encontró la cuenta: $accountName";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando banacc";
    }
}

// COMANDO /banchar - Banear/desbanear personaje
function procesarBanChar($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Comando incorrecto\n\n📝 Uso: /banchar <personaje> <valor>\n\nEjemplos:\n/banchar Usuario1 1 (banear)\n/banchar Usuario1 0 (desbanear)";
    }
    
    [$cmd, $charName, $banValue] = $parts;
    $banValue = intval($banValue);
    
    if ($banValue !== 0 && $banValue !== 1) {
        return "❌ Valor de ban inválido\n\nDebe ser 0 (desbanear) o 1 (banear)";
    }
    
    try {
        $stmt = $sqlserver->prepare("UPDATE Character SET CtlCode = ? WHERE Name = ?");
        $stmt->execute([$banValue, $charName]);
        
        if ($stmt->rowCount() > 0) {
            $status = $banValue === 1 ? "baneado" : "desbaneado";
            return "✅ Acción exitosa\n\n👤 Personaje: $charName\n📊 Estado: $status";
        } else {
            return "❌ Error\n\nNo se encontró el personaje: $charName";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando banchar";
    }
}

// COMANDO /banguild - Banear guild completo
function procesarBanGuild($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje, 4);
    
    if (count($parts) !== 4) {
        return "❌ Formato Incorrecto\n\n📝 Uso: /banguild <guild> <tipo> <acción>\n\n🎯 Tipo de Ban:\n• acc = Ban a la cuenta completa\n• char = Ban solo al personaje\n\n⚡ Acción:\n• 1 = Banear  • 0 = Desbanear\n\n📋 Ejemplos:\n/banguild UNITED acc 1\n/banguild DRAGON char 0";
    }
    
    [$cmd, $guildName, $banType, $banAction] = $parts;
    $banType = strtolower(trim($banType));
    $banAction = intval($banAction);
    
    if ($banType !== 'acc' && $banType !== 'char') {
        return "❌ Error de Tipo\n\n📋 Tipo debe ser:\n• acc = Banear cuenta completa\n• char = Banear solo personaje\n\n🔄 Intenta nuevamente";
    }
    
    if ($banAction !== 0 && $banAction !== 1) {
        return "❌ Error de Validación\n\n📋 Acción debe ser:\n• 0 = Desbanear\n• 1 = Banear\n\n🔄 Intenta nuevamente";
    }
    
    try {
        $stmt = $sqlserver->prepare("SELECT COUNT(*) FROM GuildMember WHERE G_Name = ?");
        $stmt->execute([$guildName]);
        
        if ($stmt->fetchColumn() == 0) {
            return "❌ Guild No Encontrado\n\n🏰 Guild: $guildName\n📝 Verifica que el nombre del guild sea correcto";
        }
        
        $stmt = $sqlserver->prepare("SELECT Name FROM GuildMember WHERE G_Name = ?");
        $stmt->execute([$guildName]);
        $members = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        if (empty($members)) {
            return "❌ Sin Miembros\n\n🏰 Guild: $guildName\n👥 No se encontraron miembros en este guild";
        }
        
        $successCount = 0;
        $failedCount = 0;
        $actionText = $banAction === 1 ? "baneados" : "desbaneados";
        $banTarget = $banType === 'acc' ? 'cuentas' : 'personajes';
        
        foreach ($members as $charName) {
            if ($banType === 'acc') {
                $stmt = $sqlserver->prepare("SELECT AccountID FROM Character WHERE Name = ?");
                $stmt->execute([$charName]);
                $accountId = $stmt->fetchColumn();
                
                if ($accountId) {
                    $stmt = $sqlserver->prepare("UPDATE MEMB_INFO SET bloc_code = ? WHERE memb___id = ?");
                    if ($stmt->execute([$banAction, $accountId])) {
                        $successCount++;
                    } else {
                        $failedCount++;
                    }
                } else {
                    $failedCount++;
                }
            } else {
                $stmt = $sqlserver->prepare("UPDATE Character SET CtlCode = ? WHERE Name = ?");
                if ($stmt->execute([$banAction, $charName])) {
                    $successCount++;
                } else {
                    $failedCount++;
                }
            }
        }
        
        $totalMembers = count($members);
        
        return "🏰 ¡Acción de Ban en Guild!\n\n🏷️ Guild: $guildName\n🎯 Objetivo: $banTarget\n📊 Resultados:\n✅ Exitosos: $successCount\n❌ Fallidos: $failedCount\n👥 Total: $totalMembers\n\n🎯 ¡$successCount $banTarget $actionText del guild!";
        
    } catch (PDOException $e) {
        return "Error al procesar el comando banguild";
    }
}

// COMANDO /ban - Banear IP
function procesarBan($mensaje, $sqlserver, $usuario) {
    $parts = preg_split('/\s+/', $mensaje, 3);
    
    if (count($parts) < 2) {
        return "❌ Comando incorrecto\n\n📝 Uso: /ban <IP/usuario/personaje> [razón]\n\nEjemplos:\n/ban 192.168.1.1\n/ban Usuario1 Spam\n/ban Personaje1 Hack\n\n⚠️ Importante: Para que los bans funcionen correctamente, es necesario descargar e instalar el software IPFirewall en el servidor";
    }
    
    [$cmd, $searchTerm] = $parts;
    $banReason = isset($parts[2]) ? trim($parts[2]) : 'No especificada';
    $targetIp = null;
    $foundType = '';
    $foundName = '';
    
    // Verificar si es IP directamente
    if (filter_var($searchTerm, FILTER_VALIDATE_IP)) {
        $targetIp = $searchTerm;
        $foundType = 'IP';
        $foundName = $searchTerm;
    } else {
        // Buscar IP por usuario o personaje
        $stmt = $sqlserver->prepare("
            SELECT mi.memb___id, ms.IP, 'ACCOUNT' as type
            FROM MEMB_INFO mi
            LEFT JOIN MEMB_STAT ms ON mi.memb___id = ms.memb___id
            WHERE mi.memb___id = ? AND ms.IP IS NOT NULL
            
            UNION ALL
            
            SELECT mi.memb___id, ms.IP, 'CHARACTER' as type
            FROM Character c
            INNER JOIN MEMB_INFO mi ON c.AccountID = mi.memb___id
            LEFT JOIN MEMB_STAT ms ON mi.memb___id = ms.memb___id
            WHERE c.Name = ? AND ms.IP IS NOT NULL
        ");
        $stmt->execute([$searchTerm, $searchTerm]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$result || !$result['IP']) {
            return "🚫 No se encontró IP asociada\n\nNo se pudo encontrar una IP para: $searchTerm";
        }
        
        $targetIp = $result['IP'];
        $foundType = $result['type'];
        $foundName = $result['type'] === 'ACCOUNT' ? $result['memb___id'] : $searchTerm;
    }
    
    try {
        // Verificar si existe tabla BanWPP
        $stmt = $sqlserver->prepare("SELECT CASE WHEN EXISTS(SELECT 1 FROM sys.tables WHERE name = 'BanWPP') THEN 1 ELSE 0 END as table_exists");
        $stmt->execute();
        $tableExists = $stmt->fetchColumn();
        
        // Crear tabla si no existe
        if (!$tableExists) {
            $sqlserver->exec("
                CREATE TABLE BanWPP (
                    ID int IDENTITY(1,1) PRIMARY KEY,
                    IP nvarchar(45) NOT NULL,
                    BannedBy nvarchar(50),
                    BanDate datetime DEFAULT GETDATE(),
                    Reason nvarchar(255),
                    IsActive bit DEFAULT 1
                )
            ");
            $sqlserver->exec("CREATE INDEX IX_BanWPP_IP ON BanWPP (IP)");
            $sqlserver->exec("CREATE INDEX IX_BanWPP_Active ON BanWPP (IsActive)");
        }
        
        // Verificar si ya está baneada
        $stmt = $sqlserver->prepare("SELECT ID, BanDate, Reason FROM BanWPP WHERE IP = ? AND IsActive = 1");
        $stmt->execute([$targetIp]);
        $existingBan = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existingBan) {
            $banDate = date('d/m/Y H:i', strtotime($existingBan['BanDate']));
            return "⚠️ IP ya está baneada\n\n🌐 IP: $targetIp\n📅 Fecha del ban: $banDate\n📝 Razón: {$existingBan['Reason']}";
        }
        
        // Insertar ban
        $stmt = $sqlserver->prepare("INSERT INTO BanWPP (IP, BannedBy, Reason) VALUES (?, ?, ?)");
        $stmt->execute([$targetIp, $usuario, $banReason]);
        
        $respuesta = "✅ BAN APLICADO EXITOSAMENTE\n\n🌐 IP baneada: $targetIp\n";
        
        if ($foundType !== 'IP') {
            $typeLabel = $foundType === 'ACCOUNT' ? 'Usuario' : 'Personaje';
            $respuesta .= "🎯 Origen: $foundName ($typeLabel)\n";
        }
        
        $respuesta .= "📝 Razón: $banReason\n📅 Fecha: " . date('d/m/Y H:i');
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando ban";
    }
}

// COMANDO /unban - Desbanear IP
function procesarUnban($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 2) {
        return "❌ Comando incorrecto\n\n📝 Uso: /unban <IP/usuario/personaje>\n\nEjemplos:\n/unban 192.168.1.1\n/unban Usuario1\n/unban Personaje1";
    }
    
    [$cmd, $searchTerm] = $parts;
    $targetIp = null;
    $foundType = '';
    $foundName = '';
    
    // Verificar si es IP directamente
    if (filter_var($searchTerm, FILTER_VALIDATE_IP)) {
        $targetIp = $searchTerm;
        $foundType = 'IP';
        $foundName = $searchTerm;
    } else {
        // Buscar IP por usuario o personaje
        $stmt = $sqlserver->prepare("
            SELECT mi.memb___id, ms.IP, 'ACCOUNT' as type
            FROM MEMB_INFO mi
            LEFT JOIN MEMB_STAT ms ON mi.memb___id = ms.memb___id
            WHERE mi.memb___id = ? AND ms.IP IS NOT NULL
            
            UNION ALL
            
            SELECT mi.memb___id, ms.IP, 'CHARACTER' as type
            FROM Character c
            INNER JOIN MEMB_INFO mi ON c.AccountID = mi.memb___id
            LEFT JOIN MEMB_STAT ms ON mi.memb___id = ms.memb___id
            WHERE c.Name = ? AND ms.IP IS NOT NULL
        ");
        $stmt->execute([$searchTerm, $searchTerm]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$result || !$result['IP']) {
            return "🚫 No se encontró IP asociada\n\nNo se pudo encontrar una IP para: $searchTerm";
        }
        
        $targetIp = $result['IP'];
        $foundType = $result['type'];
        $foundName = $result['type'] === 'ACCOUNT' ? $result['memb___id'] : $searchTerm;
    }
    
    try {
        // Verificar si existe tabla BanWPP
        $stmt = $sqlserver->prepare("SELECT CASE WHEN EXISTS(SELECT 1 FROM sys.tables WHERE name = 'BanWPP') THEN 1 ELSE 0 END as table_exists");
        $stmt->execute();
        $tableExists = $stmt->fetchColumn();
        
        if (!$tableExists) {
            return "🚫 Tabla BanWPP no existe\n\nNo se han registrado bans previamente";
        }
        
        // Verificar si está baneada
        $stmt = $sqlserver->prepare("SELECT ID, BanDate, Reason, BannedBy FROM BanWPP WHERE IP = ? AND IsActive = 1");
        $stmt->execute([$targetIp]);
        $existingBan = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$existingBan) {
            return "⚠️ IP no está baneada\n\n🌐 IP: $targetIp\n\n💡 Esta IP no tiene bans activos";
        }
        
        // Desactivar ban
        $stmt = $sqlserver->prepare("UPDATE BanWPP SET IsActive = 0 WHERE IP = ? AND IsActive = 1");
        $stmt->execute([$targetIp]);
        
        $banDate = date('d/m/Y H:i', strtotime($existingBan['BanDate']));
        
        $respuesta = "✅ UNBAN APLICADO EXITOSAMENTE\n\n🌐 IP desbaneada: $targetIp\n";
        
        if ($foundType !== 'IP') {
            $typeLabel = $foundType === 'ACCOUNT' ? 'Usuario' : 'Personaje';
            $respuesta .= "🎯 Origen: $foundName ($typeLabel)\n";
        }
        
        $respuesta .= "📅 Ban original: $banDate\n📝 Razón del ban: {$existingBan['Reason']}\n📅 Fecha unban: " . date('d/m/Y H:i');
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando unban";
    }
}

// COMANDO /banco - Ver banco de ítems
function procesarBanco($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 2) {
        return "📝 Formato: /banco <cuenta/personaje>\n\n💡 Ejemplos:\n• /banco Usuario1\n• /banco MiPersonaje";
    }
    
    [$cmd, $searchTerm] = $parts;
    
    try {
        // Buscar AccountID
        $accountId = null;
        $searchedByCharacter = false;
        
        $stmt = $sqlserver->prepare("SELECT memb___id FROM MEMB_INFO WHERE memb___id = ?");
        $stmt->execute([$searchTerm]);
        
        if ($stmt->fetchColumn()) {
            $accountId = $searchTerm;
        } else {
            $stmt = $sqlserver->prepare("SELECT AccountID FROM Character WHERE Name = ?");
            $stmt->execute([$searchTerm]);
            $accountId = $stmt->fetchColumn();
            
            if (!$accountId) {
                return "🚫 No se encontró la cuenta ni el personaje '$searchTerm'";
            }
            $searchedByCharacter = true;
        }
        
        $respuesta = "📦 Banco de ítems\n\n👤 Cuenta: $accountId\n";
        if ($searchedByCharacter) {
            $respuesta .= "🎮 Personaje: $searchTerm\n";
        }
        $respuesta .= "\n";
        
        // Intentar CustomItemBank
        try {
            $stmt = $sqlserver->prepare("SELECT ItemIndex, ItemCount FROM CustomItemBank WHERE AccountID = ? AND ItemCount > 0");
            $stmt->execute([$accountId]);
            $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($items)) {
                $itemNames = [
                    6159 => "Jewel of Chaos",
                    7181 => "Jewel of Bless",
                    7182 => "Jewel of Soul",
                    7184 => "Jewel of Life",
                    7190 => "Jewel of Creation",
                    7199 => "Jewel of Guardian",
                    7210 => "Jewel of Harmony",
                    7211 => "Lower refining stone",
                    7212 => "Higher refining stone",
                    7209 => "Gemstone"
                ];
                
                $respuesta .= "🔍 Ítems encontrados:\n";
                foreach ($items as $item) {
                    $itemName = $itemNames[$item['ItemIndex']] ?? "Ítem desconocido ({$item['ItemIndex']})";
                    $respuesta .= "➤ $itemName: {$item['ItemCount']}\n";
                }
                
                return $respuesta;
            }
        } catch (PDOException $e) {
            // CustomItemBank no existe, intentar Kosh
        }
        
        // Intentar sistema Kosh
        try {
            $koshColumns = ['Chaos', 'Bless', 'Souls', 'Lifes', 'Creations'];
            $existingColumns = [];
            
            foreach ($koshColumns as $column) {
                $stmt = $sqlserver->prepare("SELECT $column FROM Character WHERE AccountID = ? LIMIT 1");
                $stmt->execute([$accountId]);
                if ($stmt->errorCode() === '00000') {
                    $existingColumns[] = $column;
                }
            }
            
            if (!empty($existingColumns)) {
                $columnsStr = implode(', ', $existingColumns);
                
                if ($searchedByCharacter) {
                    $stmt = $sqlserver->prepare("SELECT Name, $columnsStr FROM Character WHERE Name = ?");
                    $stmt->execute([$searchTerm]);
                } else {
                    $stmt = $sqlserver->prepare("SELECT Name, $columnsStr FROM Character WHERE AccountID = ? ORDER BY Name");
                    $stmt->execute([$accountId]);
                }
                
                $koshData = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if (!empty($koshData)) {
                    $koshItemNames = [
                        'Chaos' => 'Jewel of Chaos',
                        'Bless' => 'Jewel of Bless',
                        'Souls' => 'Jewel of Soul',
                        'Lifes' => 'Jewel of Life',
                        'Creations' => 'Jewel of Creation'
                    ];
                    
                    $respuesta .= "🔍 Ítems por personaje:\n\n";
                    
                    foreach ($koshData as $char) {
                        $hasItems = false;
                        $charItems = "";
                        
                        foreach ($existingColumns as $column) {
                            $amount = intval($char[$column] ?? 0);
                            if ($amount > 0) {
                                $itemName = $koshItemNames[$column] ?? $column;
                                $charItems .= "➤ $itemName: $amount\n";
                                $hasItems = true;
                            }
                        }
                        
                        if ($hasItems) {
                            $respuesta .= "🎮 {$char['Name']}\n$charItems\n";
                        }
                    }
                    
                    return $respuesta;
                }
            }
        } catch (PDOException $e) {
            // Sistema Kosh no disponible
        }
        
        return $respuesta . "🔍 No se encontraron ítems en el banco";
        
    } catch (PDOException $e) {
        return "Error al procesar el comando banco";
    }
}

// COMANDO /top - Rankings TOP 10
function procesarTop($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 2) {
        return "📝 Formato: /top <ítem/moneda>\n\n💡 Ejemplos:\n• /top chaos\n• /top wc\n\n🏆 Ítems: chaos, bless, souls, lifes, creations\n💰 Monedas: wc, wp, gp, ru";
    }
    
    [$cmd, $filter] = $parts;
    $filter = strtolower(trim($filter));
    
    try {
        // Verificar si es comando de monedas
        $moneyFilters = ['wc', 'wp', 'gp', 'ru'];
        
        if (in_array($filter, $moneyFilters)) {
            $moneyConfig = [
                'wc' => ['name' => 'WCoinC', 'table' => 'CashShopData', 'column' => 'WCoinC'],
                'wp' => ['name' => 'WCoinP', 'table' => 'CashShopData', 'column' => 'WCoinP'],
                'gp' => ['name' => 'Goblin Points', 'table' => 'CashShopData', 'column' => 'GoblinPoint'],
                'ru' => ['name' => 'Ruud', 'table' => 'Character', 'column' => 'RuudMoney']
            ];
            
            $config = $moneyConfig[$filter];
            
            if ($filter === 'ru') {
                try {
                    $stmt = $sqlserver->prepare("SELECT TOP 10 AccountID, SUM(RuudMoney) as Amount FROM Character WHERE RuudMoney > 0 GROUP BY AccountID ORDER BY SUM(RuudMoney) DESC");
                    $stmt->execute();
                    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
                } catch (PDOException $e) {
                    $stmt = $sqlserver->prepare("SELECT TOP 10 AccountID, SUM(RuudToken) as Amount FROM Character WHERE RuudToken > 0 GROUP BY AccountID ORDER BY SUM(RuudToken) DESC");
                    $stmt->execute();
                    $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
                }
            } else {
                $stmt = $sqlserver->prepare("SELECT TOP 10 AccountID, {$config['column']} as Amount FROM {$config['table']} WHERE {$config['column']} > 0 ORDER BY {$config['column']} DESC");
                $stmt->execute();
                $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
            }
            
            $respuesta = "🏆 TOP 10 - {$config['name']}\n💼 Ranking por cuentas\n\n";
            
            if (empty($results)) {
                $respuesta .= "🔍 No se encontraron cuentas con {$config['name']}";
            } else {
                $medals = ['🥇', '🥈', '🥉', '4️⃣', '5️⃣', '6️⃣', '7️⃣', '8️⃣', '9️⃣', '🔟'];
                foreach ($results as $i => $result) {
                    $medal = $medals[$i] ?? "#" . ($i + 1);
                    $respuesta .= "$medal {$result['AccountID']}: {$result['Amount']}\n";
                }
            }
            
            return $respuesta;
        }
        
        // Comando de ítems
        $itemFilters = [
            'chaos' => 'chaos', 'bless' => 'bless', 'soul' => 'souls', 'souls' => 'souls',
            'life' => 'lifes', 'lifes' => 'lifes', 'creation' => 'creations', 'creations' => 'creations'
        ];
        
        if (!isset($itemFilters[$filter])) {
            return "❌ Ítem no reconocido\n\nEl ítem '$filter' no es válido\n\n📋 Ítems disponibles:\nchaos, bless, souls, lifes, creations\n\n💰 Monedas disponibles:\nwc, wp, gp, ru";
        }
        
        $normalizedFilter = $itemFilters[$filter];
        
        // Intentar CustomItemBank
        try {
            $custombankItems = [
                'chaos' => [6159, 'Jewel of Chaos'],
                'bless' => [7181, 'Jewel of Bless'],
                'souls' => [7182, 'Jewel of Soul'],
                'lifes' => [7184, 'Jewel of Life'],
                'creations' => [7190, 'Jewel of Creation']
            ];
            
            if (isset($custombankItems[$normalizedFilter])) {
                [$itemIndex, $itemName] = $custombankItems[$normalizedFilter];
                
                $stmt = $sqlserver->prepare("SELECT TOP 10 c.Name, cib.ItemCount FROM CustomItemBank cib INNER JOIN Character c ON cib.AccountID = c.AccountID WHERE cib.ItemIndex = ? AND cib.ItemCount > 0 ORDER BY cib.ItemCount DESC");
                $stmt->execute([$itemIndex]);
                $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                $respuesta = "🏆 TOP 10 - $itemName\n🎮 Ranking por personajes\n\n";
                
                if (empty($results)) {
                    $respuesta .= "🔍 No se encontraron personajes con este ítem";
                } else {
                    $medals = ['🥇', '🥈', '🥉', '4️⃣', '5️⃣', '6️⃣', '7️⃣', '8️⃣', '9️⃣', '🔟'];
                    foreach ($results as $i => $result) {
                        $medal = $medals[$i] ?? "#" . ($i + 1);
                        $respuesta .= "$medal {$result['Name']}: {$result['ItemCount']}\n";
                    }
                }
                
                return $respuesta;
            }
        } catch (PDOException $e) {
            // CustomItemBank no existe, intentar Kosh
        }
        
        // Intentar sistema Kosh
        try {
            $koshItems = [
                'chaos' => ['Chaos', 'Jewel of Chaos'],
                'bless' => ['Bless', 'Jewel of Bless'],
                'souls' => ['Souls', 'Jewel of Soul'],
                'lifes' => ['Lifes', 'Jewel of Life'],
                'creations' => ['Creations', 'Jewel of Creation']
            ];
            
            if (isset($koshItems[$normalizedFilter])) {
                [$column, $itemName] = $koshItems[$normalizedFilter];
                
                $stmt = $sqlserver->prepare("SELECT TOP 10 Name, $column as ItemCount FROM Character WHERE $column > 0 ORDER BY $column DESC");
                $stmt->execute();
                $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                $respuesta = "🏆 TOP 10 - $itemName\n🎮 Ranking por personajes\n\n";
                
                if (empty($results)) {
                    $respuesta .= "🔍 No se encontraron personajes con este ítem";
                } else {
                    $medals = ['🥇', '🥈', '🥉', '4️⃣', '5️⃣', '6️⃣', '7️⃣', '8️⃣', '9️⃣', '🔟'];
                    foreach ($results as $i => $result) {
                        $medal = $medals[$i] ?? "#" . ($i + 1);
                        $respuesta .= "$medal {$result['Name']}: {$result['ItemCount']}\n";
                    }
                }
                
                return $respuesta;
            }
        } catch (PDOException $e) {
            return "❌ No se detectó un sistema de banco compatible en esta base de datos";
        }
        
        return "❌ No se pudo procesar la solicitud para el ítem '$filter'";
        
    } catch (PDOException $e) {
        return "Error al procesar el comando top";
    }
}

// COMANDO /rastrear - Versión completa con detección VPN/Proxy y cache inteligente
function procesarRastrear($mensaje, $sqlserver) {
    // Sanitización del mensaje
    $mensaje = str_replace("\xc2\xa0", ' ', $mensaje);
    $mensaje = trim(preg_replace('/\s+/', ' ', $mensaje));
    $parts = explode(' ', $mensaje);

    if (count($parts) !== 2) {
        return "❌ Formato incorrecto\n\nUso correcto:\n/rastrear <IP o Personaje o Cuenta>\n\nEjemplos:\n/rastrear 192.168.1.1\n/rastrear LordStroll\n/rastrear freddyun";
    }

    $search_term = trim($parts[1]);
    
    // Cargar el archivo JSON de consultas
    $consultas_file = __DIR__ . '/consultas.json';
    $consultas = file_exists($consultas_file) ? json_decode(file_get_contents($consultas_file), true) : [];

    try {
        // Determinar tipo de búsqueda (IP, nombre de personaje o AccountID)
        if (filter_var($search_term, FILTER_VALIDATE_IP)) {
            // Es una IP
            $ip_address = $search_term;
            $account_id = null;
            $search_type = 'ip';
        } else {
            // Verificar si es un AccountID
            $stmt_check_account = $sqlserver->prepare("SELECT TOP 1 memb___id, IP, ConnectTM, DisConnectTM, ConnectStat 
                                FROM MEMB_STAT 
                                WHERE memb___id = ? 
                                AND IP IS NOT NULL 
                                ORDER BY ConnectTM DESC");
            $stmt_check_account->execute([$search_term]);
            $account_result = $stmt_check_account->fetch(PDO::FETCH_ASSOC);

            if ($account_result) {
                // Es un AccountID válido
                $ip_address = $account_result['IP'];
                $account_id = $search_term;
                $search_type = 'account';
            } else {
                // Intentar búsqueda por nombre de personaje
                $stmt_character = $sqlserver->prepare("SELECT AccountID, Name FROM Character WHERE Name = ?");
                $stmt_character->execute([$search_term]);
                $character_result = $stmt_character->fetch(PDO::FETCH_ASSOC);
                
                if (!$character_result) {
                    return "❌ No encontrado\nNo se encontró ningún personaje ni cuenta con el identificador '$search_term'";
                }
                
                $account_id = $character_result['AccountID'];
                $search_type = 'character';
                
                // Obtener la IP usando el AccountID
                $stmt_ip = $sqlserver->prepare("SELECT TOP 1 IP, memb___id, ConnectTM, DisConnectTM, ConnectStat 
                          FROM MEMB_STAT 
                          WHERE memb___id = ? 
                          AND IP IS NOT NULL 
                          ORDER BY ConnectTM DESC");
                $stmt_ip->execute([$account_id]);
                $ip_result = $stmt_ip->fetch(PDO::FETCH_ASSOC);
                
                if (!$ip_result || empty($ip_result['IP'])) {
                    return "❌ IP no encontrada\nNo se encontró IP asociada al personaje '$search_term'";
                }
                
                $ip_address = $ip_result['IP'];
            }
        }

        // Mapeo de países con emojis
        $country_names = [
            'CL' => 'Chile 🇨🇱',
            'PE' => 'Perú 🇵🇪',
            'MX' => 'México 🇲🇽',
            'AR' => 'Argentina 🇦🇷',
            'BO' => 'Bolivia 🇧🇴',
            'CO' => 'Colombia 🇨🇴',
            'EC' => 'Ecuador 🇪🇨',
            'VE' => 'Venezuela 🇻🇪',
            'BR' => 'Brasil 🇧🇷',
            'UY' => 'Uruguay 🇺🇾',
            'PY' => 'Paraguay 🇵🇾',
            'NI' => 'Nicaragua 🇳🇮',
            'BS' => 'Bahamas 🇧🇸',
            'TT' => 'Trinidad y Tobago 🇹🇹',
            'US' => 'Estados Unidos 🇺🇸',
            'CA' => 'Canadá 🇨🇦',
            'VN' => 'Vietnam 🇻🇳',
            'GB' => 'Reino Unido 🇬🇧',
            'FR' => 'Francia 🇫🇷',
            'AW' => 'Aruba 🇦🇼',
            'ES' => 'España 🇪🇸',
            'LV' => 'Letonia 🇱🇻',
            'DE' => 'Alemania 🇩🇪',
            'IT' => 'Italia 🇮🇹'
        ];

        // Mapeo de números de mapa a nombres
        $map_names = [
            0 => "Lorencia", 1 => "Dungeon", 2 => "Devias", 3 => "Noria", 4 => "Lost Tower", 6 => "Arena",
            7 => "Atlans", 8 => "Tarkan", 9 => "Devil Square 1", 10 => "Icarus", 11 => "Blood Castle 1",
            12 => "Blood Castle 2", 13 => "Blood Castle 3", 14 => "Blood Castle 4", 15 => "Blood Castle 5",
            16 => "Blood Castle 6", 17 => "Blood Castle 7", 18 => "Chaos Castle 1", 19 => "Chaos Castle 2",
            20 => "Chaos Castle 3", 21 => "Chaos Castle 4", 22 => "Chaos Castle 5", 23 => "Chaos Castle 6",
            24 => "Kalima 1", 25 => "Kalima 2", 26 => "Kalima 3", 27 => "Kalima 4", 28 => "Kalima 5",
            29 => "Kalima 6", 30 => "Castle Siege", 31 => "Land of Trials", 32 => "Devil Square 2",
            33 => "Aida", 34 => "Crywolf", 36 => "Kalima 7", 37 => "Kanturu 1", 38 => "Kanturu 2",
            39 => "Kanturu 3", 40 => "Silent", 41 => "Barracks", 42 => "Refuge", 45 => "Illusion Temple 1",
            46 => "Illusion Temple 2", 47 => "Illusion Temple 3", 48 => "Illusion Temple 4", 49 => "Illusion Temple 5",
            50 => "Illusion Temple 6", 51 => "Elbeland", 52 => "Blood Castle 8", 53 => "Chaos Castle 7",
            56 => "Swamp of Calmness", 57 => "Raklion 1", 58 => "Raklion 2", 62 => "Santa Town",
            63 => "Vulcanus", 64 => "Duel Arena", 65 => "Double Goer 1", 66 => "Double Goer 2",
            67 => "Double Goer 3", 68 => "Double Goer 4", 69 => "Imperial Guardian 1", 70 => "Imperial Guardian 2",
            71 => "Imperial Guardian 3", 72 => "Imperial Guardian 4", 79 => "Loren Market", 80 => "Karutan 1",
            81 => "Karutan 2"
        ];

        // Inicializar la estructura de la IP si no existe
        if (!isset($consultas[$ip_address])) {
            $consultas[$ip_address] = [
                'count' => 0,
                'accounts' => [],
                'characters' => [],
                'queries' => []
            ];
        }

        // Registrar consulta
        $consultas[$ip_address]['count']++;
        $consultas[$ip_address]['queries'][] = date('Y-m-d H:i:s');

        if ($account_id && !in_array($account_id, $consultas[$ip_address]['accounts'])) {
            $consultas[$ip_address]['accounts'][] = $account_id;
        }

        if ($search_type === 'character' && !in_array($search_term, $consultas[$ip_address]['characters'])) {
            $consultas[$ip_address]['characters'][] = $search_term;
        }

        // Guardar el archivo JSON actualizado
        file_put_contents($consultas_file, json_encode($consultas, JSON_PRETTY_PRINT));

        // Consulta mejorada para obtener todos los usuarios con esa IP incluyendo ConnectStat
        $stmt_rastrear = $sqlserver->prepare("SELECT DISTINCT m.memb___id, m.ConnectTM, m.DisConnectTM, m.ConnectStat 
                        FROM MEMB_STAT m
                        WHERE m.IP = ? 
                        ORDER BY m.ConnectTM DESC");
        $stmt_rastrear->execute([$ip_address]);
        $results = $stmt_rastrear->fetchAll(PDO::FETCH_ASSOC);

        // Sistema de cache inteligente para API de proxycheck.io
        $cache_file = __DIR__ . '/proxy_cache.json';
        $daily_usage_file = __DIR__ . '/proxy_usage.json';
        $max_daily_queries = 950; // Límite diario (dejamos margen)
        $cache_duration_days = 7; // Cache por 7 días (las IPs proxy/VPN no cambian frecuentemente)
        
        // Cargar cache existente
        $proxy_cache = [];
        if (file_exists($cache_file)) {
            $proxy_cache = json_decode(file_get_contents($cache_file), true) ?: [];
        }
        
        // Cargar uso diario
        $daily_usage = [];
        if (file_exists($daily_usage_file)) {
            $daily_usage = json_decode(file_get_contents($daily_usage_file), true) ?: [];
        }
        
        // Obtener fecha actual
        $today = date('Y-m-d');
        
        // Resetear contador si es un nuevo día
        if (!isset($daily_usage[$today])) {
            $daily_usage = [$today => 0]; // Limpiar días anteriores
        }
        
        // Verificar si la IP está en cache y es válida
        $use_cache = false;
        $cache_expired = false;
        
        if (isset($proxy_cache[$ip_address])) {
            $cached_data = $proxy_cache[$ip_address];
            $cache_date = new DateTime($cached_data['cached_date']);
            $current_date = new DateTime();
            $diff_days = $current_date->diff($cache_date)->days;
            
            if ($diff_days < $cache_duration_days) {
                $use_cache = true;
            } else {
                $cache_expired = true;
            }
        }
        
        // Obtener país y información de proxy/VPN
        $country_name = "País no disponible 🌎";
        $vpn_proxy_status = "❓ Error";
        $abuse_status = "✅ NO"; // Por defecto asumir NO abuso si no se puede verificar
        
        if ($use_cache) {
            // Usar datos del cache
            $cached_data = $proxy_cache[$ip_address];
            $country_name = $cached_data['country'] ?? $country_name;
            $vpn_proxy_status = $cached_data['vpn_proxy_status'] ?? $vpn_proxy_status;
            $abuse_status = $cached_data['abuse_status'] ?? "✅ NO"; // Valor por defecto si no existe
            
        } elseif ($daily_usage[$today] < $max_daily_queries) {
            // Hacer consulta a la API solo si no hemos alcanzado el límite
            try {
                $api_key = "0t5531-02hw11-87o558-n98159";
                $proxy_api_url = "https://proxycheck.io/v2/$ip_address?key=$api_key&vpn=1&asn=1&risk=1&port=1&seen=1&days=7&tag=mu_server";
                $context = stream_context_create([
                    'http' => [
                        'timeout' => 5,
                        'user_agent' => 'MU Server Admin Tool'
                    ]
                ]);
                $proxy_response = @file_get_contents($proxy_api_url, false, $context);
                
                if ($proxy_response !== false) {
                    $proxy_data = json_decode($proxy_response, true);
                    
                    // Incrementar contador de uso diario
                    $daily_usage[$today]++;
                    
                    // Verificar si la respuesta es válida
                    if (isset($proxy_data['status']) && $proxy_data['status'] === 'ok' && isset($proxy_data[$ip_address])) {
                        $ip_info = $proxy_data[$ip_address];
                        
                        // Obtener país
                        if (isset($ip_info['country'])) {
                            $country_name = $ip_info['country'];
                            if (isset($ip_info['isocode'])) {
                                $country_code = $ip_info['isocode'];
                                $country_name = $country_names[$country_code] ?? $country_name;
                            }
                        }
                        
                        // Determinar si es VPN/Proxy (simplificado)
                        $is_vpn_proxy = false;
                        $is_abuse = false;
                        
                        // Verificar proxy
                        if (isset($ip_info['proxy']) && strtolower($ip_info['proxy']) === 'yes') {
                            $is_vpn_proxy = true;
                        }
                        
                        // Verificar tipo de conexión
                        if (isset($ip_info['type'])) {
                            $type = strtoupper($ip_info['type']);
                            if (in_array($type, ['VPN', 'PUB', 'WEB', 'TOR'])) {
                                $is_vpn_proxy = true;
                            }
                        }
                        
                        // Verificar abuso (risk score a partir de 5%)
                        if (isset($ip_info['risk'])) {
                            $risk_score = intval($ip_info['risk']);
                            if ($risk_score >= 5) { // Cambio: >= 5 para detectar desde 5%
                                $is_abuse = true;
                            }
                            // Si existe el campo risk pero es bajo, definitivamente NO es abuso
                            $abuse_status = $is_abuse ? "🚫 SÍ" : "✅ NO";
                        } else {
                            // Si no hay información de risk, asumir que no hay abuso
                            $abuse_status = "✅ NO";
                        }
                        
                        // Crear status de VPN/Proxy
                        $vpn_proxy_status = $is_vpn_proxy ? "🚫 SÍ" : "✅ NO";
                        
                        // Guardar en cache
                        $proxy_cache[$ip_address] = [
                            'country' => $country_name,
                            'vpn_proxy_status' => $vpn_proxy_status,
                            'abuse_status' => $abuse_status,
                            'cached_date' => date('Y-m-d H:i:s'),
                            'api_response' => $ip_info
                        ];
                        
                    } else {
                        // Error en API de proxycheck, usar fallback para país
                        $vpn_proxy_status = "❓ Error API";
                        $abuse_status = "❓ Error API";
                        $country_name = getCountryFallbackRastrear($ip_address, $country_names);
                    }
                    
                    // Guardar archivos de cache y uso
                    file_put_contents($cache_file, json_encode($proxy_cache, JSON_PRETTY_PRINT));
                    file_put_contents($daily_usage_file, json_encode($daily_usage, JSON_PRETTY_PRINT));
                    
                } else {
                    $vpn_proxy_status = "❓ Sin respuesta";
                    $abuse_status = "❓ Sin respuesta";
                    $country_name = getCountryFallbackRastrear($ip_address, $country_names);
                }
                
            } catch (Exception $e) {
                $vpn_proxy_status = "❓ Error";
                $abuse_status = "❓ Error";
                $country_name = getCountryFallbackRastrear($ip_address, $country_names);
            }
            
        } else {
            // Límite diario alcanzado
            $vpn_proxy_status = "⏳ Límite alcanzado";
            $abuse_status = "⏳ Límite alcanzado";
            $country_name = getCountryFallbackRastrear($ip_address, $country_names);
            
            // Si tenemos datos caducados, usarlos como fallback
            if ($cache_expired && isset($proxy_cache[$ip_address])) {
                $cached_data = $proxy_cache[$ip_address];
                $country_name = $cached_data['country'] ?? $country_name;
                $vpn_proxy_status = ($cached_data['vpn_proxy_status'] ?? "❓ No disponible") . " (Cache)";
                $abuse_status = ($cached_data['abuse_status'] ?? "✅ NO") . " (Cache)";
            }
        }

        // Verificar si hay resultados
        if (empty($results)) {
            return "❌ IP no encontrada\nNo se encontraron usuarios con la IP $ip_address\n🌎 País: $country_name";
        }

        // Construir respuesta con el nuevo formato
        $user_count = count($results);
        
        $response = "🔍 Rastreo";
        $response .= match($search_type) {
            'character' => " por personaje",
            'account' => " por cuenta",
            default => ""
        };
        $response .= "\n\n";
        
        if ($search_type == 'character') {
            $response .= "👤 Personaje: $search_term\n";
            $response .= "🏠 Cuenta: $account_id\n";
        } elseif ($search_type == 'account') {
            $response .= "👤 Cuenta: $account_id\n";
        }
        
        $response .= "📍 IP: $ip_address\n";
        $response .= "🛡️ VPN/Proxy: $vpn_proxy_status\n";
        $response .= "⚠️ Abuso: $abuse_status\n";
        $response .= "🌎 País: $country_name\n";
        $response .= "👥 Cuentas encontradas: $user_count\n\n";
        
        foreach ($results as $index => $row) {
            $response .= "━━━━━ Cuenta #" . ($index + 1) . " ━━━━━\n";
            $response .= "📝 Cuenta: " . $row['memb___id'] . "\n";
            
            // Estado de conexión actual
            $connection_status = $row['ConnectStat'] == 1 ? "🟢 Conectado" : "🔴 Desconectado";
            $response .= "📊 Estado: $connection_status\n\n";
            
            $response .= "🔓 Último ingreso:\n• " . formatDateRastrear($row['ConnectTM']) . "\n\n";
            
            if (!empty($row['DisConnectTM']) && $row['DisConnectTM'] != '0000-00-00 00:00:00') {
                $response .= "🔒 Última salida:\n• " . formatDateRastrear($row['DisConnectTM']) . "\n\n";
            }
            
            // Obtener personajes y ubicaciones
            $characters = getCharactersWithLocationsRastrear($sqlserver, $row['memb___id'], $map_names);
            
            if (!empty($characters)) {
                $response .= "🎮 Personajes:\n";
                
                foreach ($characters as $char) {
                    $response .= "• {$char['name']} - {$char['location']}\n";
                }
            } else {
                $response .= "🎮 Personajes:\n• Sin personajes\n";
            }
            
            $response .= "\n";
        }
        
        $response .= "🕐 Las fechas y horas corresponden al servidor.\n";
        $response .= "⚠️ Abuso: IPs con historial de actividades maliciosas.";
        
        return $response;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando rastrear: " . $e->getMessage();
    }
}

// Función auxiliar para obtener país como fallback
function getCountryFallbackRastrear($ip, $country_names) {
    try {
        $country_api_url = "https://api.country.is/$ip";
        $country_response = @file_get_contents($country_api_url);
        if ($country_response !== false) {
            $country_data = json_decode($country_response, true);
            $country_code = $country_data['country'] ?? null;
            if ($country_code) {
                return $country_names[$country_code] ?? "País desconocido 🌎";
            }
        }
    } catch (Exception $e) {
        // Silenciar error
    }
    return "País no disponible 🌎";
}

// Función para formatear fechas
function formatDateRastrear($datetime) {
    if (empty($datetime) || $datetime == '0000-00-00 00:00:00') {
        return "No registrado";
    }
    try {
        $date = new DateTime($datetime);
        return $date->format('d/m/Y - h:i A');
    } catch (Exception $e) {
        return "Fecha inválida";
    }
}

// Función para obtener personajes y sus ubicaciones
function getCharactersWithLocationsRastrear($sqlserver, $account_id, $map_names) {
    try {
        $stmt_chars = $sqlserver->prepare("SELECT Name, MapNumber, MapPosX, MapPosY 
                                       FROM Character 
                                       WHERE AccountID = ? 
                                       ORDER BY Name");
        $stmt_chars->execute([$account_id]);
        $characters = $stmt_chars->fetchAll(PDO::FETCH_ASSOC);
        
        $char_info = [];
        foreach ($characters as $char) {
            $map_name = $map_names[$char['MapNumber']] ?? "Mapa {$char['MapNumber']}";
            $char_info[] = [
                'name' => $char['Name'],
                'location' => "$map_name ({$char['MapPosX']}, {$char['MapPosY']})"
            ];
        }
        
        return $char_info;
    } catch (PDOException $e) {
        return [];
    }
}

// COMANDO /global - Búsqueda global en múltiples servidores
function procesarGlobal($mensaje, $pdo) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 2) {
        return "❌ Formato incorrecto\n\nUso: /global <IP o Personaje o Cuenta>\n\nEjemplos:\n/global 192.168.1.1\n/global LordStroll\n/global freddyun";
    }
    
    [$cmd, $searchTerm] = $parts;
    
    // Implementación básica - en un entorno real necesitarías acceso a múltiples bases de datos
    try {
        $results = [];
        
        if (filter_var($searchTerm, FILTER_VALIDATE_IP)) {
            $stmt = $pdo->prepare("SELECT DISTINCT TOP 10 memb___id FROM MEMB_STAT WHERE IP = ?");
            $stmt->execute([$searchTerm]);
            $results = $stmt->fetchAll(PDO::FETCH_COLUMN);
        } else {
            $stmt = $pdo->prepare("SELECT memb___id FROM MEMB_INFO WHERE memb___id = ?");
            $stmt->execute([$searchTerm]);
            $account = $stmt->fetchColumn();
            
            if ($account) {
                $results = [$account];
            }
        }
        
        $respuesta = "🌐 Búsqueda Global Completada\n\n🔍 Término: $searchTerm\n\n📊 Resultados:\n";
        
        if (!empty($results)) {
            $respuesta .= "• Servidores con coincidencias: 1\n";
            $respuesta .= "• Total de cuentas: " . count($results) . "\n\n";
            $respuesta .= "Cuentas:\n";
            foreach ($results as $account) {
                $respuesta .= "• $account\n";
            }
        } else {
            $respuesta .= "❌ No se encontraron resultados";
        }
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando global";
    }
}

// COMANDO /guildposts - Ver avisos de guilds
function procesarGuildPosts($sqlserver) {
    try {
        $stmt = $sqlserver->prepare("SELECT G_Name, G_Master, G_Notice, G_Count FROM Guild WHERE G_Notice IS NOT NULL AND G_Notice != '' AND G_Notice != 'NULL' ORDER BY G_Master, G_Name");
        $stmt->execute();
        $guildNotices = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($guildNotices)) {
            return "📢 Avisos de Guilds\n\n❌ Sin Avisos Configurados\n\n💡 Ningún guild tiene avisos establecidos en este momento";
        }
        
        $masterNotices = [];
        $totalGuilds = 0;
        
        foreach ($guildNotices as $guild) {
            $master = $guild['G_Master'];
            if (!isset($masterNotices[$master])) {
                $masterNotices[$master] = [];
            }
            $masterNotices[$master][] = $guild;
            $totalGuilds++;
        }
        
        $totalMasters = count($masterNotices);
        
        $respuesta = "📢 Posts de Guilds\n\n📊 Resumen:\n👑 Masters con avisos: $totalMasters\n🏰 Guilds con avisos: $totalGuilds\n\n";
        
        $mastersShown = 0;
        $maxMasters = 5;
        
        foreach ($masterNotices as $master => $guilds) {
            if ($mastersShown >= $maxMasters) {
                $remaining = $totalMasters - $maxMasters;
                $respuesta .= "... y $remaining masters más\n\n💡 Usa /guildnotice <guild> para ver avisos específicos";
                break;
            }
            
            $respuesta .= "👑 Master: $master\n━━━━━━━━━━━━━━━━━━━━\n";
            
            foreach ($guilds as $guild) {
                $guildName = $guild['G_Name'];
                $guildCount = $guild['G_Count'];
                $notice = trim($guild['G_Notice']);
                
                if (strlen($notice) > 60) {
                    $notice = substr($notice, 0, 57) . "...";
                }
                
                $respuesta .= "🏰 $guildName ($guildCount miembros)\n📝 \"$notice\"\n\n";
            }
            
            $mastersShown++;
        }
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando guildposts";
    }
}

// COMANDO /whichguild - Ver en qué guild está un personaje
function procesarWhichGuild($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 2) {
        return "❌ Formato Incorrecto\n\n📝 Uso: /whichguild <personaje>\n\n📋 Ejemplo:\n/whichguild PlayerName\n\n🔍 Muestra en qué guild está el personaje";
    }
    
    [$cmd, $charName] = $parts;
    
    try {
        $stmt = $sqlserver->prepare("SELECT COUNT(*) FROM Character WHERE Name = ?");
        $stmt->execute([$charName]);
        
        if ($stmt->fetchColumn() == 0) {
            return "❌ Personaje No Encontrado\n\n👤 Personaje: $charName\n📝 Verifica que el nombre sea correcto";
        }
        
        $stmt = $sqlserver->prepare("SELECT G_Name FROM GuildMember WHERE Name = ?");
        $stmt->execute([$charName]);
        $guildInfo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $respuesta = "🔍 Verificación de Guild\n\n👤 Personaje: $charName\n\n";
        
        if (!$guildInfo) {
            $respuesta .= "🏠 Estado del Guild:\n❌ Sin Guild\n\n💡 Este personaje no pertenece a ningún guild actualmente";
        } else {
            $guildName = $guildInfo['G_Name'];
            
            $stmt = $sqlserver->prepare("SELECT COUNT(*) FROM GuildMember WHERE G_Name = ?");
            $stmt->execute([$guildName]);
            $memberCount = $stmt->fetchColumn();
            
            $stmt = $sqlserver->prepare("SELECT G_Master, G_Notice FROM Guild WHERE G_Name = ?");
            $stmt->execute([$guildName]);
            $guildDetails = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $respuesta .= "🏰 Información del Guild:\n🏷️ Nombre: $guildName\n👥 Miembros: $memberCount\n";
            
            if ($guildDetails && $guildDetails['G_Master']) {
                $respuesta .= "👑 Master: {$guildDetails['G_Master']}\n";
            }
            
            if (isset($guildDetails['G_Notice']) && !empty($guildDetails['G_Notice']) && 
                trim($guildDetails['G_Notice']) !== '' && $guildDetails['G_Notice'] !== 'NULL') {
                $noticePreview = strlen($guildDetails['G_Notice']) > 60 ? 
                               substr($guildDetails['G_Notice'], 0, 57) . "..." : 
                               $guildDetails['G_Notice'];
                $respuesta .= "\n📢 Aviso del Guild:\n📝 \"$noticePreview\"\n";
            }
            
            $respuesta .= "\n✅ El personaje es miembro activo del guild";
        }
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando whichguild";
    }
}

// COMANDO /fixonline - Desconectar usuarios fantasma
function procesarFixOnline($sqlserver) {
    try {
        $stmt = $sqlserver->prepare("UPDATE MEMB_STAT SET ConnectStat = 0 WHERE ConnectStat = 1");
        
        if ($stmt->execute()) {
            return "✅ Usuarios fantasma desconectados\n\nTodos los usuarios han sido marcados como desconectados";
        } else {
            return "❌ Error al actualizar el estado de los usuarios";
        }
        
    } catch (PDOException $e) {
        return "Error al procesar el comando fixonline";
    }
}

// Función auxiliar para formatear tamaños
function formatSizeClearLog($sizeMB) {
    if ($sizeMB >= 1024 * 1024) {
        return number_format($sizeMB / (1024 * 1024), 2) . ' TB';
    } elseif ($sizeMB >= 1024) {
        return number_format($sizeMB / 1024, 2) . ' GB';
    } else {
        return number_format($sizeMB, 2) . ' MB';
    }
}

// COMANDO /clearlog - Limpiar logs de base de datos
function procesarClearLog($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    $showList = false;
    if (count($parts) >= 2 && strtolower(trim($parts[1])) === 'list') {
        $showList = true;
    }
    
    try {
        $currentDb = $sqlserver->query("SELECT DB_NAME()")->fetchColumn() ?: 'Base de datos actual';
        
        $stmt = $sqlserver->query("SELECT name, CAST(size*8.0/1024 AS DECIMAL(10,2)) as size_mb FROM sys.database_files WHERE type = 1");
        $logFiles = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($logFiles)) {
            return "❌ No se encontraron archivos de log\n🗄️ Base de datos: $currentDb";
        }
        
        if ($showList) {
            $respuesta = "📋 Archivos de Log Disponibles\n🗄️ Base de datos: $currentDb\n\n";
            foreach ($logFiles as $index => $file) {
                $formattedSize = formatSizeClearLog($file['size_mb']);
                $respuesta .= "📄 " . ($index + 1) . ". {$file['name']} ($formattedSize)\n";
            }
            $respuesta .= "\n💡 Usa /clearlog para limpiar automáticamente";
            return $respuesta;
        }
        
        $logFileName = $logFiles[0]['name'];
        $originalSize = formatSizeClearLog($logFiles[0]['size_mb']);
        
        $stmt1 = $sqlserver->prepare("ALTER DATABASE [$currentDb] SET RECOVERY SIMPLE");
        $stmt1->execute();
        
        $stmt2 = $sqlserver->prepare("DBCC SHRINKFILE ([$logFileName], 1)");
        $stmt2->execute();
        
        $stmt3 = $sqlserver->prepare("ALTER DATABASE [$currentDb] SET RECOVERY FULL");
        $stmt3->execute();
        
        return "✅ ¡Limpieza Completada!\n🗄️ Base de datos: $currentDb\n📋 Archivo: $logFileName\n📊 Tamaño original: $originalSize";
        
    } catch (PDOException $e) {
        return "❌ Error en la Limpieza\nNo se pudo completar la operación.\nContacta al administrador.";
    } catch (Exception $e) {
        return "❌ Error en la Limpieza\nNo se pudo completar la operación.\nContacta al administrador.";
    }
}

// COMANDO /index - Calcular índice
function procesarIndex($mensaje) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "📝 Formato: /index <Sección> <Index>\n\n💡 Ejemplo:\n• /index 7 14\n\nCalcula: (Sección × 512) + Index";
    }
    
    [$cmd, $number1, $number2] = $parts;
    $number1 = intval($number1);
    $number2 = intval($number2);
    
    if ($number1 < 0 || $number2 < 0) {
        return "❌ Ambos números deben ser enteros positivos";
    }
    
    $result = ($number1 * 512) + $number2;
    
    return "✅ Cálculo realizado exitosamente\n\n🔢 Número 1: $number1\n🔢 Número 2: $number2\n✖️ Multiplicación: $number1 * 512 = " . ($number1 * 512) . "\n➕ Suma: " . ($number1 * 512) . " + $number2 = $result\n\n🎯 Resultado final: $result";
}

// COMANDO /indexrm - Obtener sección e índice
function procesarIndexRm($mensaje) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 2) {
        return "📝 Formato: /indexrm <Número>\n\n💡 Ejemplo:\n• /indexrm 6159\n\nObtiene Sección e Index del número";
    }
    
    [$cmd, $number] = $parts;
    $number = intval($number);
    
    if ($number < 0) {
        return "❌ El número debe ser un entero positivo";
    }
    
    $number1 = intdiv($number, 512);
    $number2 = $number % 512;
    
    return "✅ Cálculo realizado exitosamente\n\n🔢 Número proporcionado: $number\n✖️ División: $number ÷ 512 = $number1 (cociente)\n➕ Residuo: $number2\n\n🎯 Resultado: $number1 $number2";
}

// COMANDO /setsaldo - Establecer saldo
function procesarSetSaldo($mensaje, $sqlserver) {
    $parts = preg_split('/\s+/', $mensaje);
    
    if (count($parts) !== 3) {
        return "❌ Formato Incorrecto\n\n📝 Uso: /setsaldo <Cuenta|Personaje> <Monto>\n\n📋 Ejemplo:\n/setsaldo user123 500\n/setsaldo MiPJ 300";
    }
    
    [$cmd, $inputId, $amount] = $parts;
    $amount = intval($amount);
    
    if ($amount <= 0) {
        return "❌ El monto debe ser mayor a 0";
    }
    
    try {
        $finalAccountId = null;
        
        $stmt = $sqlserver->prepare("SELECT AccountID FROM Saldo WHERE AccountID = ?");
        $stmt->execute([$inputId]);
        
        if ($stmt->fetchColumn() !== false) {
            $finalAccountId = $inputId;
        } else {
            $stmt = $sqlserver->prepare("SELECT AccountID FROM Character WHERE Name = ?");
            $stmt->execute([$inputId]);
            $accountId = $stmt->fetchColumn();
            
            if ($accountId === false) {
                return "❌ Cuenta o personaje '$inputId' no encontrado";
            } else {
                $finalAccountId = $accountId;
            }
        }
        
        $stmt = $sqlserver->prepare("SELECT USD FROM Saldo WHERE AccountID = ?");
        $stmt->execute([$finalAccountId]);
        $currentUsd = $stmt->fetchColumn();
        
        if ($currentUsd !== false) {
            $newUsd = $currentUsd + $amount;
            $stmt = $sqlserver->prepare("UPDATE Saldo SET USD = ? WHERE AccountID = ?");
            $stmt->execute([$newUsd, $finalAccountId]);
        } else {
            $stmt = $sqlserver->prepare("INSERT INTO Saldo (AccountID, USD) VALUES (?, ?)");
            $stmt->execute([$finalAccountId, $amount]);
            $newUsd = $amount;
        }
        
        $respuesta = "💰 Saldo actualizado\n\n👤 Cuenta: $finalAccountId\n";
        
        if ($finalAccountId !== $inputId) {
            $respuesta .= "🎮 Personaje: $inputId\n";
        }
        
        $respuesta .= "➕ Monto agregado: $amount USD\n💵 Saldo total: $newUsd USD";
        
        return $respuesta;
        
    } catch (PDOException $e) {
        return "Error al procesar el comando setsaldo";
    }
}

// COMANDO /ayuda - Lista completa de comandos disponibles
function procesarAyuda() {
    return "📋 COMANDOS DISPONIBLES\n━━━━━━━━━━━━━━━━━━━━\n\n👑 GESTIÓN DE VIP\n\n▫️ /setvip [ID/PJ] [Nivel] [Días]\n   Asignar VIP (cuenta desconectada)\n\n▫️ /setvipf [ID/PJ] [Nivel] [Días]\n   Asignar VIP (cuenta conectada)\n\n▫️ /setvipguild [Guild] [Nivel] [Días]\n   Asignar VIP a todo un guild\n\n▫️ /setviprm [ID/PJ]\n   Remover VIP de una cuenta\n\n▫️ /setvipall [Nivel] [Días]\n   VIP masivo a cuentas sin VIP válido\n\n▫️ /infovip\n   Estadísticas de usuarios VIP\n\n━━━━━━━━━━━━━━━━━━━━\n\n💰 GESTIÓN DE MONEDAS\n\n▫️ /setcoin [ID/PJ] [Tipo] [Cantidad]\n   Añadir monedas (desconectado)\n\n▫️ /setcoinf [ID/PJ] [Tipo] [Cantidad]\n   Añadir monedas (conectado)\n\n▫️ /setcoinrm [ID/PJ] [Tipo] [Cantidad]\n   Remover monedas\n\n▫️ /setcoinall [Tipo] [Cantidad]\n   Monedas masivas a todas las cuentas\n\n━━━━━━━━━━━━━━━━━━━━\n\n🦸 GESTIÓN DE PERSONAJES\n\n▫️ /reset [PJ] [Cantidad]\n   Añadir resets (desconectado)\n\n▫️ /resetf [PJ] [Cantidad]\n   Añadir resets (conectado)\n\n▫️ /resetrm [PJ] [Cantidad]\n   Remover resets\n\n▫️ /mreset [PJ] [Cantidad]\n   Añadir master resets (desconectado)\n\n▫️ /mresetf [PJ] [Cantidad]\n   Añadir master resets (conectado)\n\n▫️ /mresetrm [PJ] [Cantidad]\n   Remover master resets\n\n▫️ /level [PJ] [Nivel]\n   Modificar nivel (desconectado)\n\n▫️ /zen [PJ] [Cantidad]\n   Añadir zen al personaje\n\n▫️ /setstats [PJ] [Valor]\n   Todas las stats al mismo valor\n\n▫️ /s [PJ] [Valor] - Fuerza\n▫️ /a [PJ] [Valor] - Agilidad\n▫️ /v [PJ] [Valor] - Vitalidad\n▫️ /e [PJ] [Valor] - Energía\n▫️ /c [PJ] [Valor] - Command (DL)\n\n━━━━━━━━━━━━━━━━━━━━\n\n⭐ BATTLE PASS\n\n▫️ /battlepass [PJ] [Cantidad]\n   Añadir estrellas\n\n▫️ /battlerm [PJ] [Cantidad]\n   Remover estrellas\n\n━━━━━━━━━━━━━━━━━━━━\n\n🔒 GESTIÓN DE BANS\n\n▫️ /banacc [ID] [0|1]\n   Banear/desbanear cuenta\n\n▫️ /banchar [PJ] [0|1]\n   Banear/desbanear personaje\n\n▫️ /banguild [Guild] [acc|char] [0|1]\n   Banear/desbanear guild completo\n\n▫️ /ban [IP/ID/PJ] [razón]\n   Banear IP con razón opcional\n\n▫️ /unban [IP/ID/PJ]\n   Desbanear IP\n\n━━━━━━━━━━━━━━━━━━━━\n\n🔑 GESTIÓN DE CUENTAS\n\n▫️ /password [ID]\n   Obtener contraseña de cuenta\n\n▫️ /changepw [ID] [Dificultad|Contraseña]\n   Cambiar contraseña (1-5 o personalizada)\n\n━━━━━━━━━━━━━━━━━━━━\n\nℹ️ INFORMACIÓN Y RASTREO\n\n▫️ /info [ID/PJ]\n   Información completa de cuenta\n\n▫️ /infopj [PJ]\n   Información detallada del personaje\n\n▫️ /rastrear [IP/ID/PJ]\n   Rastreo completo de conexiones\n\n▫️ /global [IP/ID/PJ]\n   Búsqueda en múltiples servidores\n\n▫️ /online\n   Estado del servidor y jugadores\n\n▫️ /banco [ID/PJ]\n   Ver banco de ítems\n\n▫️ /top [ítem|moneda]\n   Rankings TOP 10\n\n━━━━━━━━━━━━━━━━━━━━\n\n🏰 GESTIÓN DE GUILDS\n\n▫️ /guildposts\n   Ver avisos de todos los guilds\n\n▫️ /whichguild [PJ]\n   Ver en qué guild está un personaje\n\n━━━━━━━━━━━━━━━━━━━━\n\n🛠️ UTILIDADES Y ADMINISTRACIÓN\n\n▫️ /fixonline\n   Desconectar usuarios fantasma\n\n▫️ /clearlog [list]\n   Limpiar logs de BD o ver archivos\n\n▫️ /index [Sección] [Index]\n   Calcular: (Sección × 512) + Index\n\n▫️ /indexrm [Número]\n   Obtener Sección e Index del número\n\n▫️ /setsaldo [ID/PJ] [Monto]\n   Establecer saldo USD\n\n▫️ /ayuda\n   Mostrar esta lista de comandos\n\n━━━━━━━━━━━━━━━━━━━━\n\n📝 REFERENCIAS RÁPIDAS\n\n🔢 Tipos de Moneda:\n  1️⃣ WCoinC  2️⃣ WCoinP\n  3️⃣ GoblinPoints  4️⃣ Ruud\n\n⭐ Niveles VIP:\n  1️⃣ Bronze  2️⃣ Silver  3️⃣ Gold\n\n🏆 Ítems para TOP:\n  chaos, bless, souls, lifes, creations\n  guardians, harmonys, gemstones\n  lowstones, highstones\n\n💰 Monedas para TOP:\n  wc, wp, gp, ru\n\n🎯 Estados de Comandos:\n  🔴 Requiere desconectado\n  🟢 Funciona conectado\n  🔵 Solo información\n\n💡 Sistemas de Banco Compatibles:\n  • CustomItemBank (Louis)\n  • Kosh (Character columns)\n  • MSPRO (XTR_JewelStore)\n\n🔧 Sistemas de Ruud:\n  • RuudMoney (Character)\n  • RuudToken (Character)";
}
?>